const { genereteToken } = require("../../config/auth/jwt");
const nodemailer = require("nodemailer");
const path = require("path");
const https = require("https");
const fs = require("fs");
const bcrypt = require("bcrypt");
const { validarCNPJ } = require("../../util/validarCNPJ");
const { validarCPF } = require("../../util/validarCPF");

class RecebimentosRepository {
  #db = require("../../config/db");

  constructor() {}

  /**
   *
   * @param {*} data
   * @returns
   */
  register(data) {
    return new Promise(async (resolve, reject) => {
      try {
        const resultado =
          data.cnpj_cpf.length === 11
            ? validarCPF(data.cnpj_cpf)
            : data.cnpj_cpf.length === 14
              ? validarCNPJ(data.cnpj_cpf)
              : null;

        if (!resultado) {
          return reject({
            error: true,
            message: "CNPJ ou CPF inválido!",
            code: 422,
          });
        }

        const hashedPassword = await bcrypt.hash(data.senha, 10);

        this.#db.query(
          `INSERT INTO recebimentos 
         (nome_empresa, nome, celular, email, ramo_atividade, senha, cnpj_cpf)
         VALUES (?, ?, ?, ?, ?, ?, ?)`,
          [
            data.nome_empresa,
            data.nome,
            data.celular || "",
            data.email || "",
            data.ramo_atividade || "",
            hashedPassword,
            data.cnpj_cpf,
          ],
          async (error, response) => {
            if (error) {
              console.error(error);
              return reject({
                error: true,
                message: "Erro ao cadastrar recebimento",
                code: 500,
              });
            }

            // INSERT na tabela users
            this.#db.query(
              `INSERT INTO users (nome, cnpj_cpf, senha, role, trueId)
             SELECT nome_empresa, cnpj_cpf, senha, 'recebimento', id
             FROM recebimentos
            `,
              (error) => {
                if (error) {
                  console.error(error);
                  return reject({
                    error: true,
                    message: "Erro ao cadastrar usuário recebimento",
                    code: 500,
                  });
                }

                // UPDATE com o id_user
                this.#db.query(
                  `UPDATE recebimentos a
                 INNER JOIN users u ON a.cnpj_cpf = u.cnpj_cpf
                 SET a.id_user = u.id
                 WHERE u.role = 'recebimento' 
                   AND a.cnpj_cpf = ?`,
                  [data.cnpj_cpf],
                  (error) => {
                    if (error) {
                      console.error(error);
                      return reject({
                        error: true,
                        message: "Erro ao atualizar recebimento",
                        code: 500,
                      });
                    }

                    return resolve({
                      success: true,
                      message: "Recebimento cadastrado com sucesso!",
                      code: 200,
                    });
                  },
                );
              },
            );
          },
        );
      } catch (err) {
        console.error("Erro ao cadastrar recebimento ", err);
        return reject({
          error: true,
          message: "Erro ao cadastrar recebimento",
          code: 500,
        });
      }
    });
  }

  /**
   * Lista todas as lojas
   */
  getRecebimentos() {
    return new Promise((resolve, reject) => {
      try {
        this.#db.query(
          `SELECT id, nome_empresa, nome, celular, email, ramo_atividade, role, cnpj_cpf
         FROM recebimentos`,
          [],
          (error, recebimentos) => {
            if (error) {
              console.error("Erro ao buscar recebimentos ", error);
              return reject({
                error: true,
                message: "Erro ao buscar recebimentos",
                code: 500,
              });
            }
            return resolve(recebimentos);
          },
        );
      } catch (error) {
        console.error("Erro ao buscar recebimentos ", error);
        return reject({
          error: true,
          message: "Erro ao buscar recebimentos",
          code: 500,
        });
      }
    });
  }

  /**
   * Busca apenas um fornecedor
   * @param {*} id
   * @returns
   */
  getUmRecebimento(id) {
    return new Promise((resolve, reject) => {
      try {
        this.#db.query(
          `SELECT id, nome_empresa, nome, celular, email, ramo_atividade, role, cnpj_cpf
         FROM recebimentos
         WHERE id = ?`,
          [id],
          (error, results) => {
            if (error) {
              console.error("Erro ao buscar recebimento ", error);
              return reject({
                error: true,
                message: "Erro ao buscar recebimento",
                code: 500,
              });
            }

            if (results.length === 0) {
              return reject({
                error: true,
                message: "Recebimento não encontrado",
                code: 404,
              });
            }

            return resolve(results[0]);
          },
        );
      } catch (error) {
        console.error("Erro ao buscar fornecedor ", error);
        return reject({
          error: true,
          message: "Erro ao buscar fornecedor",
          code: 500,
        });
      }
    });
  }

  /**
   * Deleta uma loja
   * @param {*} id
   * @returns
   */
  delete(id) {
    return new Promise((resolve, reject) => {
      this.#db.query("DELETE FROM recebimentos WHERE id = ?", [id], (error) => {
        if (error) {
          console.error("Erro ao deletar recebimentos ", error);
          return reject({
            error: true,
            message: "Este recebimentos possui vínculos!",
            code: 409,
          });
        }
        return resolve({
          success: true,
          message: "recebimentos deletado com sucesso!",
          code: 200,
        });
      });
    });
  }

  /**
   * Atualiza um fornecedor
   * @param {Object} data
   * @returns Mensagem referente ao status da operação
   */
  update(data) {
    return new Promise(async (resolve, reject) => {
      try {
        const fields = [];
        const values = [];

        if (data.nome_empresa) {
          fields.push("nome_empresa = ?");
          values.push(data.nome_empresa);
        }
        if (data.nome) {
          fields.push("nome = ?");
          values.push(data.nome);
        }
        if (data.telefone) {
          fields.push("telefone = ?");
          values.push(data.telefone);
        }
        if (data.email) {
          fields.push("email = ?");
          values.push(data.email);
        }
        if (data.ramo_atividade) {
          fields.push("ramo_atividade = ?");
          values.push(data.ramo_atividade);
        }
        if (hashedPassword) {
          fields.push("senha = ?");
          values.push(hashedPassword);
        }

        if (fields.length === 0) {
          return reject({
            error: true,
            message: "Nenhum dado para atualizar",
            code: 422,
          });
        }

        values.push(data.id);

        const query = `
        UPDATE recebimentos
        SET ${fields.join(", ")}
        WHERE id = ?
      `;

        this.#db.query(query, values, (error, response) => {
          if (error) {
            console.error(error);
            return reject({
              error: true,
              message: "Erro ao atualizar recebimentos",
              code: 500,
            });
          }

          if (response.affectedRows === 0) {
            return reject({
              error: true,
              message: "Recebimento não encontrado",
              code: 404,
            });
          }

          return resolve({
            success: true,
            message: "recebimentos atualizado com sucesso!",
            code: 200,
          });
        });
      } catch (error) {
        console.error(error);
        return reject({
          error: true,
          message: "Erro ao atualizar recebimentos",
          code: 500,
        });
      }
    });
  }
}

module.exports = RecebimentosRepository;
