const { S3Client, PutObjectCommand } = require("@aws-sdk/client-s3");

class PlantasRepository {
  #db = require("../../config/db");
  #region = process.env.AWS_S3_REGION;

  constructor() {
    this.s3Client = new S3Client({
      region: process.env.AWS_S3_REGION,
      credentials: {
        accessKeyId: process.env.AWS_ACCESS_KEY_ID,
        secretAccessKey: process.env.AWS_SECRET_ACCESS_KEY,
      },
    });

    this.S3_BUCKET = process.env.AWS_S3_BUCKET;
  }

  /**
   * Cadastra uma planta
   */
  async register(data, file) {
  return new Promise(async (resolve, reject) => {
    try {

      let fileUrl = null;

      // Se veio arquivo, envia para o S3
      if (file) {
        const fileName = `${Date.now()}-${file.originalname}`;

        const upload = new PutObjectCommand({
          Bucket: this.S3_BUCKET,
          Key: fileName,
          Body: file.buffer,
          ContentType: file.mimetype
        });

        await this.s3Client.send(upload);

        fileUrl = `https://${this.S3_BUCKET}.s3.${this.#region}.amazonaws.com/${fileName}`;
      }

      // Insere no banco
      this.#db.query(
        `INSERT INTO plantas (id_empresa, url_arquivo)
         VALUES (?, ?)`,
        [data.id_empresa, fileUrl],
        (error, response) => {
          if (error) {
            console.error(error);
            return reject({
              error: true,
              message: "Erro ao cadastrar planta",
              code: 500
            });
          }

          return resolve({
            success: true,
            message: "Planta cadastrada com sucesso!",
            url: fileUrl,
            code: 200
          });
        }
      );

    } catch (error) {
      console.error(error);
      return reject({
        error: true,
        message: "Erro ao cadastrar planta",
        code: 500
      });
    }
  });
}


  /**
   * Lista todas plantas
   */
  getPlantas() {
    return new Promise((resolve, reject) => {
      this.#db.query(
        `SELECT id, id_empresa, url_arquivo FROM plantas`,
        [],
        (error, results) => {
          if (error) {
            console.error(error);
            return reject({ error: true, message: "Erro ao buscar plantas", code: 500 });
          }
          return resolve(results);
        }
      );
    });
  }

  /**
   * Busca uma planta pelo ID
   */
  getUmaPlanta(id) {
    return new Promise((resolve, reject) => {
      this.#db.query(
        `SELECT id, id_empresa, url_arquivo
         FROM plantas
         WHERE id = ?`,
        [id],
        (error, results) => {
          if (error) {
            console.error(error);
            return reject({ error: true, message: "Erro ao buscar planta", code: 500 });
          }

          if (results.length === 0) {
            return reject({ error: true, message: "Planta não encontrada", code: 404 });
          }

          return resolve(results[0]);
        }
      );
    });
  }

  /**
   * Deleta uma planta
   */
  delete(id) {
    return new Promise((resolve, reject) => {
      this.#db.query(
        "DELETE FROM plantas WHERE id = ?",
        [id],
        (error, response) => {
          if (error) {
            console.error(error);
            return reject({ error: true, message: "Erro ao deletar planta", code: 500 });
          }

          if (response.affectedRows === 0) {
            return reject({ error: true, message: "Planta não encontrada", code: 404 });
          }

          return resolve({ success: true, message: "Planta deletada com sucesso!", code: 200 });
        }
      );
    });
  }

  /**
   * Atualiza informações da planta
   */
  async update(data, file) {
  return new Promise(async (resolve, reject) => {
    try {
      const fields = [];
      const values = [];

      if (data.id_empresa) {
        fields.push("id_empresa = ?");
        values.push(data.id_empresa);
      }

      let novaUrl = null;

      if (file) {
        const fileName = `${Date.now()}-${file.originalname}`;

        const upload = new PutObjectCommand({
          Bucket: this.S3_BUCKET,
          Key: fileName,
          Body: file.buffer,
          ContentType: file.mimetype
        });

        await this.s3Client.send(upload);

        novaUrl = `https://${this.S3_BUCKET}.s3.${this.#region}.amazonaws.com/${fileName}`;

        fields.push("url_arquivo = ?");
        values.push(novaUrl);
      }

      if (fields.length === 0) {
        return reject({ error: true, message: "Nenhum dado para atualizar", code: 422 });
      }

      values.push(data.id);

      const query = `
        UPDATE plantas
        SET ${fields.join(", ")}
        WHERE id = ?
      `;

      this.#db.query(query, values, (error, response) => {
        if (error) {
          console.error(error);
          return reject({
            error: true,
            message: "Erro ao atualizar planta",
            code: 500
          });
        }

        if (response.affectedRows === 0) {
          return reject({
            error: true,
            message: "Planta não encontrada",
            code: 404
          });
        }

        return resolve({
          success: true,
          message: "Planta atualizada com sucesso!",
          url: novaUrl || data.url_arquivo,
          code: 200
        });
      });

    } catch (error) {
      console.error(error);
      return reject({
        error: true,
        message: "Erro ao atualizar planta",
        code: 500
      });
    }
  });
}


  /**
   * Upload do arquivo da planta para o S3
   */
  async uploadPlanta(file, id_empresa) {
    return new Promise(async (resolve, reject) => {
      try {
        if (!file) {
          return reject({ error: true, message: "Arquivo não enviado", code: 400 });
        }

        const fileName = `${Date.now()}-${file.originalname}`;

        const upload = new PutObjectCommand({
          Bucket: this.S3_BUCKET,
          Key: fileName,
          Body: file.buffer,
          ContentType: file.mimetype
        });

        await this.s3Client.send(upload);

        const url = `https://${this.S3_BUCKET}.s3.${this.#region}.amazonaws.com/${fileName}`;

        // Salva no banco
        this.#db.query(
          `INSERT INTO plantas (id_empresa, url_arquivo) VALUES (?, ?)`,
          [id_empresa, url],
          (error, result) => {
            if (error) {
              console.error(error);
              return reject({ error: true, message: "Erro ao salvar planta", code: 500 });
            }

            return resolve({ success: true, url, code: 200 });
          }
        );

      } catch (error) {
        console.error(error);
        return reject({ error: true, message: "Erro ao fazer upload da planta", code: 500 });
      }
    });
  }
}

module.exports = PlantasRepository;
