const { sendMessage } = require("../../util/gzappy");
const { S3Client, PutObjectCommand } = require("@aws-sdk/client-s3");
const FornecedoresRepository = require("../fornecedores/fornecedores.repository");
const CompradoresRepository = require("../compradores/compradores.repository");
const { enviarEmail } = require("../../util/enviarEmail");
const { parseQuantity } = require("../../util/separaNumeroUnidadeMedida");

class PedidosRepository {
  #db = require("../../config/db");
  #region = process.env.AWS_S3_REGION;
  #fornecedores;
  #compradores;

  constructor(
    fornecedoresRepository = new FornecedoresRepository(),
    compradoresRepository = new CompradoresRepository(),
  ) {
    this.#fornecedores = fornecedoresRepository;
    this.#compradores = compradoresRepository;
    this.s3Client = new S3Client({
      region: process.env.AWS_S3_REGION,
      credentials: {
        accessKeyId: process.env.AWS_ACCESS_KEY_ID,
        secretAccessKey: process.env.AWS_SECRET_ACCESS_KEY,
      },
    });
    this.S3_BUCKET = process.env.AWS_S3_BUCKET;
    this.region = process.env.AWS_S3_REGION;
  }

  register(data, files) {
    return new Promise((resolve, reject) => {
      try {
        console.log("FILES:", files);
        console.log("ITEMS:", data.itens);
        // 1. Tratamento do JSON (caso venha como string do Postman)
        if (data.itens && typeof data.itens === "string") {
          try {
            data.itens = JSON.parse(data.itens);
          } catch (e) {
            return reject({
              error: true,
              message: "JSON de itens inválido",
              code: 400,
            });
          }
        }

        // 2. Função auxiliar de Upload (definida aqui dentro para acessar 'this')
        const uploadToS3 = async (file) => {
          if (!file) return null;

          const fileName = `${Date.now()}_${file.originalname.replace(
            /\s/g,
            "_",
          )}`;
          // Nota: Certifique-se que this.#region está definido na sua classe, senão use a string direta
          const imageUrl = `https://${this.S3_BUCKET}.s3.${
            this.#region || "us-east-1"
          }.amazonaws.com/${fileName}`;

          const command = new PutObjectCommand({
            Bucket: this.S3_BUCKET,
            Key: fileName,
            Body: file.buffer,
            ContentType: file.mimetype,
            // ACL: 'public-read' // Descomente se necessário
          });

          await this.s3Client.send(command);
          return imageUrl;
        };

        // 3. Inserir o Pedido (Pai)
        this.#db.query(
          `INSERT INTO pedidos (id_empresa, data_criacao, status_geral, titulo) VALUES (?, NOW(), "Pendente", ?)`,
          [data.id_empresa, data.titulo],
          (error, response) => {
            if (error) {
              console.error("Erro ao cadastrar pedido ", error);
              return reject({
                error: true,
                message: "Erro ao cadastrar pedido",
                code: 500,
              });
            }

            const id_pedido = response.insertId;

            // 4. Buscar nome da empresa (para notificação)
            this.#db.query(
              `SELECT nome FROM empresas WHERE id = ?`,
              [data.id_empresa],
              (err, empresaResults) => {
                const nomeEmpresa = empresaResults[0]?.nome || "a empresa";

                const enviarNotificacoes = () => {
                  this.#db.query(
                    `SELECT telefone, nome FROM compradores`,
                    async (err2, compradores) => {
                      if (!err2 && compradores.length > 0) {
                        for (const comprador of compradores) {
                          await sendMessage(
                            `Olá, ${comprador.nome}. Um novo pedido foi criado para a empresa ${nomeEmpresa}.

Por favor, acesse o sistema para verificar os detalhes: https://obracroissantecia.com.br

Em caso de dúvidas, entre em contato com (11) 97259-7464. Mensagem automatica, nao responder.
`,
                            comprador.telefone,
                          );
                        }
                      }
                    },
                  );
                };

                if (!data.itens || data.itens.length === 0) {
                  enviarNotificacoes();
                  return resolve({
                    success: true,
                    message: "Pedido s/ itens!",
                    code: 200,
                    id_pedido,
                  });
                }

                // 5. Função Recursiva Assíncrona para inserir Itens + Upload
                const insertItem = async (index) => {
                  if (index >= data.itens.length) {
                    enviarNotificacoes();
                    return resolve({
                      success: true,
                      message: "Pedido e itens cadastrados com sucesso!",
                      code: 200,
                      id_pedido,
                    });
                  }

                  const item = data.itens[index];
                  let urlParaSalvar = null;

                  // --- LÓGICA DE UPLOAD ---
                  if (item.midia_amostra === true) {
                    const nomeCampoArquivo = `file_${item.midia_index}`;

                    // Encontrar o arquivo correspondente no array 'files'
                    let arquivoEncontrado = null;
                    if (Array.isArray(files)) {
                      arquivoEncontrado = files.find(
                        (f) => f.fieldname === nomeCampoArquivo,
                      );
                    } else if (files && files[nomeCampoArquivo]) {
                      arquivoEncontrado = files[nomeCampoArquivo][0];
                    }

                    if (arquivoEncontrado) {
                      try {
                        // AWAIT aqui é o segredo: espera o upload antes de continuar
                        urlParaSalvar = await uploadToS3(arquivoEncontrado);
                      } catch (uploadErr) {
                        console.error("Erro no upload S3:", uploadErr);
                        // Decida se para tudo ou continua sem imagem. Aqui continuamos:
                        urlParaSalvar = null;
                      }
                    }
                  }
                  // -------------------------

                  this.#db.query(
                    `INSERT INTO pedidos_itens
                                (id_pedido, nome_material, descricao, midia_amostra,
                                data_entrega_desejada, quantidade)
                                VALUES (?, ?, ?, ?, ?, ?)`,
                    [
                      id_pedido,
                      item.nome_material,
                      item.descricao || null,
                      urlParaSalvar,
                      item.data_entrega_desejada,
                      item.quantidade,
                    ],
                    (err3) => {
                      if (err3) {
                        console.error("Erro item DB:", err3);
                        return reject({
                          error: true,
                          message: "Erro ao inserir item",
                          code: 500,
                        });
                      }
                      insertItem(index + 1);
                    },
                  );
                };

                // Inicia o loop
                insertItem(0);
              },
            );
          },
        );
      } catch (error) {
        console.error("Erro geral:", error);
        return reject({ error: true, message: "Erro interno", code: 500 });
      }
    });
  }

  inserirPedidoItens(data) {
    return new Promise((resolve, reject) => {
      try {
        this.#db.query(
          `INSERT INTO pedidos 
           (id_empresa, data_criacao, status_geral)
           VALUES (?, ?, ?)`,
          [data.id_empresa, data.data_criacao, data.status_geral],
          (error, response) => {
            if (error) {
              console.error("Erro ao cadastrar pedido ", error);
              return reject({
                error: true,
                message: "Erro ao cadastrar pedido",
                code: 500,
              });
            }

            return resolve({
              success: true,
              message: "Pedido cadastrado com sucesso!",
              code: 200,
            });
          },
        );
      } catch (error) {
        console.error("Erro ao cadastrar pedido ", error);
        return reject({
          error: true,
          message: "Erro ao cadastrar pedido",
          code: 500,
        });
      }
    });
  }

  cotacao(data) {
    return new Promise(async (resolve, reject) => {
      try {
        await new Promise((resolveUpdate, rejectUpdate) => {
          this.#db.query(
            `UPDATE pedidos_itens SET preco_acordado = null, data_entrega = null, sem_estoque = null WHERE id = ?`,
            [data.id_item],
            (error, response) => {
              if (error) return rejectUpdate(error);
              resolveUpdate();
            },
          );
        });

        const fornecedor = await this.#fornecedores.getUmFornecedor(
          data.id_fornecedor,
        );
        const fornecedorTelefoneLimpo = fornecedor.telefone.replace(/\D/g, "");
        const mensagem = `Você foi selecionado para atender a um novo pedido.

Acesse o link abaixo para conferir os itens e prazos: https://obracroissantecia.com.br

Em caso de dúvidas, entre em contato com (11) 97259-7464. Mensagem automatica, nao responder.`;

        this.#db.query(
          `INSERT INTO pedidos_itens_fornecedores 
                (id_fornecedor, id_item, status, forma_pagamento_fornecedor, preco, entrega, id_pedido)
                VALUES (?, ?, 'Aguardando cotação', NULL, NULL, NULL, ?)`,
          [data.id_fornecedor, data.id_item, data.id_pedido],
          (error, response) => {
            if (error) {
              console.error("Erro ao inserir item fornecedor ", error);
              return reject({
                error: true,
                message: "Erro ao enviar item",
                code: 500,
              });
            }

            this.#db.query(
              `SELECT id FROM pedidos_itens_fornecedores WHERE id_pedido = ? AND id_fornecedor = ?`,
              [data.id_pedido, data.id_fornecedor],
              async (error, responseVerificar) => {
                if (error) {
                  console.error("Erro na verificação", error);
                  return reject({
                    error: true,
                    message: "Erro ao verificar",
                    code: 500,
                  });
                }

                if (responseVerificar.length === 1) {
                  await sendMessage(mensagem, "55" + fornecedorTelefoneLimpo);
                }

                return resolve({
                  success: true,
                  message: "Item enviado com sucesso!",
                  code: 200,
                });
              },
            );
          },
        );
      } catch (error) {
        console.error("Erro geral: ", error);
        return reject({ error: true, message: "Erro interno", code: 500 });
      }
    });
  }

  enviar(data) {
    return new Promise((resolve, reject) => {
      try {
        this.#db.query(
          `UPDATE pedidos_itens_fornecedores 
            SET pedidos_itens_fornecedores.status = 'Em_transito'
            WHERE pedidos_itens_fornecedores.id_pedido = ? 
            AND pedidos_itens_fornecedores.id_fornecedor = ?
            AND pedidos_itens_fornecedores.status = 'Aprovado Gestor';`,
          [data.id_pedido, data.id_fornecedor],
          async (error, response) => {
            if (error) {
              console.error("Erro ao enviar pedido ", error);
              return reject({
                error: true,
                message: "Erro ao enviar pedido",
                code: 500,
              });
            }
            if (response.affectedRows === 0) {
              return reject({
                error: true,
                message: "Nenhum pedido encontrado para enviar",
                code: 404,
              });
            }

            await sendMessage(`Os itens do seu pedido saíram para entrega.

Acesse o sistema para confirmar o recebimento assim que os produtos chegarem: https://obracroissantecia.com.br

Em caso de dúvidas, entre em contato com (11) 97259-7464. Mensagem automatica, nao responder.`);
            return resolve({
              success: true,
              message: "Pedido enviado com sucesso!",
              code: 200,
            });
          },
        );
      } catch (error) {
        console.error("Erro ao enviar pedido ", error);
        return reject({
          error: true,
          message: "Erro ao enviar pedido",
          code: 500,
        });
      }
    });
  }

  updateItens(data, id) {
    return new Promise((resolve, reject) => {
      this.#db.query(
        `UPDATE pedidos_itens SET nome_material = ?, descricao = ? WHERE id = ?`,
        [data.nome_material, data.descricao, id],
        (error, response) => {
          if (error) {
            console.error("Erro ao atualizar itens do pedido ", error);
            return reject({
              error: true,
              message: "Erro ao atualizar itens do pedido",
              code: 500,
            });
          }

          return resolve({
            success: true,
            message: "Itens do pedido atualizados com sucesso!",
            code: 200,
          });
        },
      );
    });
  }

  getTransito() {
    return new Promise((resolve, reject) => {
      try {
        this.#db.query(
          `SELECT DISTINCT 
    p.id,
    p.id_empresa,
    p.status_geral,
    e.nome AS nome_empresa,
    DATE_FORMAT(p.data_criacao, '%d/%m/%Y') AS data_criacao
FROM pedidos p
LEFT JOIN empresas e 
       ON e.id = p.id_empresa
LEFT JOIN pedidos_itens pi 
       ON pi.id_pedido = p.id
LEFT JOIN pedidos_itens_fornecedores pif 
       ON pif.id_pedido = p.id
WHERE 
pif.status = 'Em_transito'
AND p.status_geral = 'Pendente';`,
          (error, response) => {
            if (error) {
              console.error("Erro ao buscar pedidos em trânsito ", error);
              return reject({
                error: true,
                message: "Erro ao listar pedidos",
                code: 500,
              });
            }

            return resolve(response);
          },
        );
      } catch (error) {
        console.error("Erro ao buscar pedidos em trânsito ", error);
        return reject({
          error: true,
          message: "Erro ao listar pedidos",
          code: 500,
        });
      }
    });
  }

  getItensTransito(id_pedido) {
    return new Promise((resolve, reject) => {
      try {
        this.#db.query(
          `SELECT 
    pi.id,
    pi.id_pedido,
    pi.nome_material,
    pi.descricao,
    DATE_FORMAT(pi.data_entrega_desejada, '%d/%m/%Y') AS data_entrega_desejada,
    pi.quantidade,
    pif.preco AS preco_acordado,
    DATE_FORMAT(pi.data_entrega, '%d/%m/%Y') AS data_entrega,
    pif.status AS status_item,
    f.nome,
    f.nome_empresa,
    f.id AS id_fornecedor
FROM pedidos_itens pi
LEFT JOIN pedidos_itens_fornecedores pif 
       ON pif.id_item = pi.id
LEFT JOIN fornecedores f 
       ON f.id = pif.id_fornecedor
WHERE 
    pi.id_pedido = ?
    AND pif.status = 'Em_transito';

`,
          [id_pedido],
          (error, response) => {
            if (error) {
              console.error("Erro ao buscar pedidos em trânsito ", error);
              return reject({
                error: true,
                message: "Erro ao listar pedidos",
                code: 500,
              });
            }

            return resolve(response);
          },
        );
      } catch (error) {
        console.error("Erro ao buscar pedidos em trânsito ", error);
        return reject({
          error: true,
          message: "Erro ao listar pedidos",
          code: 500,
        });
      }
    });
  }

  async recebimentoConfirmado(dadosJson, arquivosUpload) {
    const MAX_RETRIES = 2;
    let tentativa = 0;

    console.log("[recebimentoConfirmado] start", {
      hasJsonString: typeof dadosJson === "string",
      arquivosUploadCount: Array.isArray(arquivosUpload)
        ? arquivosUpload.length
        : 0,
    });

    const executar = async () => {
      const dados =
        typeof dadosJson === "string" ? JSON.parse(dadosJson) : dadosJson;

      const idPedido = dados.idPedido;
      const listaGrupos = dados.entregas;

      console.log("[recebimentoConfirmado] parsed", {
        idPedido,
        gruposCount: Array.isArray(listaGrupos) ? listaGrupos.length : 0,
      });

      const uploadToS3 = async (file) => {
        if (!file) return null;

        const cleanName = file.originalname.replace(/[^a-zA-Z0-9.]/g, "_");
        const fileName = `${Date.now()}_${cleanName}`;

        const imageUrl = `https://${this.S3_BUCKET}.s3.${this.region}.amazonaws.com/${fileName}`;

        await this.s3Client.send(
          new PutObjectCommand({
            Bucket: this.S3_BUCKET,
            Key: fileName,
            Body: file.buffer,
            ContentType: file.mimetype,
          }),
        );

        console.log("[recebimentoConfirmado] uploaded", {
          fileName,
          contentType: file.mimetype,
        });

        return imageUrl;
      };

      let conn;

      try {
        conn = await new Promise((res, rej) =>
          this.#db.getConnection((err, connection) =>
            err ? rej(err) : res(connection),
          ),
        );

        await new Promise((res, rej) =>
          conn.beginTransaction((err) => (err ? rej(err) : res())),
        );

        // 🔁 LOOP DOS GRUPOS
        for (let i = 0; i < listaGrupos.length; i++) {
          const grupo = listaGrupos[i];
          const obs = grupo.observacao || null;

          const [urlNf, urlEntregador, urlAssinatura] = await Promise.all([
            uploadToS3(
              arquivosUpload.find((f) => f.fieldname === `arquivos_${i}_nf`),
            ),
            uploadToS3(
              arquivosUpload.find(
                (f) => f.fieldname === `arquivos_${i}_entregador`,
              ),
            ),
            uploadToS3(
              arquivosUpload.find(
                (f) => f.fieldname === `arquivos_${i}_assinatura`,
              ),
            ),
          ]);

          // 🔥 LOOP NOS ITENS
          for (const item of grupo.itens) {
            // ignora itens não recebidos
            if (!item.quantRecebida || item.quantRecebida <= 0) continue;

            await new Promise((res, rej) =>
              conn.query(
                `
        INSERT INTO pedidos_recebimentos
        (id_pedido, id_item, id_fornecedor, nota_fiscal, foto_entregador, assinatura, observacao, data_recebimento, quant)
        VALUES (?, ?, ?, ?, ?, ?, ?, NOW(), ?)
        `,
                [
                  idPedido,
                  item.idItem, // ✅ FK correta
                  grupo.idFornecedor,
                  urlNf,
                  urlEntregador,
                  urlAssinatura,
                  obs,
                  item.quantRecebida, // ✅ quantidade correta
                ],
                (err) => (err ? rej(err) : res()),
              ),
            );
          }

          console.log("[recebimentoConfirmado] recebimento inserido", {
            idPedido,
            idFornecedor: grupo.idFornecedor,
            quant: grupo.quant,
          });

          // 🔁 LOOP DOS ITENS DO GRUPO
          for (const item of grupo.itens) {
            const quantOriginal = await new Promise((resolve, reject) => {
              conn.query(
                `SELECT quantidade FROM pedidos_itens WHERE id = ?`,
                [item.idItem],
                (err, result) =>
                  err
                    ? reject(err)
                    : console.log(
                        "[recebimentoConfirmado] quantOriginal result",
                        { result },
                      ) || resolve(result[0].quantidade),
              );
            });

            const { amount, unit } = parseQuantity(quantOriginal);
            const quantRecebida = item.quantRecebida || 0;

            console.log("[recebimentoConfirmado] item", {
              idItem: item.idItem,
              quantOriginal: amount,
              quantRecebida,
            });

            if (quantRecebida < amount) {
              await new Promise((res, rej) =>
                conn.query(
                  `UPDATE pedidos_itens SET quantidade = ? WHERE id = ?`,
                  [amount - quantRecebida, item.idItem],
                  (err) => (err ? rej(err) : res()),
                ),
              );

              await new Promise((res, rej) =>
                conn.query(
                  `UPDATE pedidos_itens_fornecedores SET status = ? WHERE id_item = ?`,
                  ["Entregue Parcial", item.idItem],
                  (err) => (err ? rej(err) : res()),
                ),
              );

              console.log("[recebimentoConfirmado] status item", {
                idItem: item.idItem,
                status: "Entregue Parcial",
              });
            } else {
              await new Promise((res, rej) =>
                conn.query(
                  `UPDATE pedidos_itens_fornecedores SET status = ? WHERE id_item = ?`,
                  ["Entregue", item.idItem],
                  (err) => (err ? rej(err) : res()),
                ),
              );

              console.log("[recebimentoConfirmado] status item", {
                idItem: item.idItem,
                status: "Entregue",
              });
            }
          }
        }

        // ✅ VERIFICAÇÃO FINAL DO PEDIDO
        const itensFornecedor = await new Promise((resolve, reject) => {
          conn.query(
            `SELECT status FROM pedidos_itens_fornecedores WHERE id_pedido = ?`,
            [idPedido],
            (err, result) => (err ? reject(err) : resolve(result)),
          );
        });

        const allEntregue = itensFornecedor.every(
          (i) => i.status === "Entregue",
        );

        if (allEntregue) {
          await new Promise((res, rej) =>
            conn.query(
              `UPDATE pedidos SET status_geral = ? WHERE id = ?`,
              ["Entregue", idPedido],
              (err) => (err ? rej(err) : res()),
            ),
          );

          console.log("[recebimentoConfirmado] pedido entregue", {
            idPedido,
          });
        }

        await new Promise((res, rej) =>
          conn.commit((err) => (err ? rej(err) : res())),
        );

        conn.release();

        console.log("[recebimentoConfirmado] commit ok", { idPedido });

        return {
          success: true,
          message: "Recebimento processado com sucesso",
          code: 200,
        };
      } catch (error) {
        if (conn) {
          await new Promise((res) => conn.rollback(() => res()));
          conn.release();
        }

        console.log("[recebimentoConfirmado] erro", {
          message: error.message,
          code: error.code,
        });
        throw error;
      }
    };

    while (tentativa <= MAX_RETRIES) {
      try {
        return await executar();
      } catch (error) {
        if (error.code === "ER_LOCK_DEADLOCK" && tentativa < MAX_RETRIES) {
          console.log("[recebimentoConfirmado] deadlock retry", {
            tentativa,
            max: MAX_RETRIES,
          });
          tentativa++;
          continue;
        }

        throw {
          error: true,
          message: error.message,
          code: error.code || 500,
        };
      }
    }
  }

  getPendenteEntrega() {
    return new Promise((resolve, reject) => {
      try {
        const sql = `
        SELECT
    p.id AS id_pedido,
    DATE_FORMAT(p.data_criacao, '%d/%m/%Y') AS data_criado,
    p.status_geral,
    p.titulo,
    e.nome AS empresa
FROM pedidos p
JOIN empresas e ON e.id = p.id_empresa
INNER JOIN pedidos_itens pi ON pi.id_pedido = p.id
INNER JOIN pedidos_itens_fornecedores pif ON pif.id_item = pi.id
WHERE p.status_geral = 'Pendente' AND pif.status = 'Entregue Parcial'

GROUP BY p.id;`;

        this.#db.query(sql, (error, results) => {
          if (error) {
            console.error("Erro ao buscar pedidos pendentes", error);
            return reject({
              error: true,
              message: "Erro ao listar pedidos",
              code: 500,
            });
          }
          resolve(results);
        });
      } catch (error) {
        console.error("Erro ao buscar pedidos pendentes", error);
        return reject({
          error: true,
          message: "Erro interno",
          code: 500,
        });
      }
    });
  }

  cancelaItemOuReenvia(itemId, data) {
    return new Promise((resolve, reject) => {
      try {
        const sqlUpdateItem = `
        UPDATE pedidos_itens_fornecedores 
        SET status = ?
        WHERE id_item = ?;
      `;

        this.#db.query(
          sqlUpdateItem,
          [data.status, itemId],
          (error, results) => {
            if (error) {
              console.error("Erro ao mudar status item", error);
              return reject({
                error: true,
                message: "Erro ao mudar status item",
                code: 500,
              });
            }

            if (results.affectedRows === 0) {
              return reject({
                error: true,
                message: "Item não encontrado",
                code: 404,
              });
            }

            // Se NÃO for cancelamento, encerra aqui
            if (data.status !== "Cancelado") {
              return resolve({
                success: true,
                message: "Item alterado com sucesso!",
                code: 200,
              });
            }

            // 1️⃣ Verifica se ainda existem itens em trânsito ou aprovados
            const sqlVerificaItens = `
          SELECT COUNT(*) AS total
          FROM pedidos_itens_fornecedores
          WHERE id_pedido = ?
          AND status IN ('Em_transito', 'Aprovado Gestor');
        `;

            this.#db.query(
              sqlVerificaItens,
              [data.id_pedido],
              (errorCheck, resultCheck) => {
                if (errorCheck) {
                  console.error(
                    "Erro ao verificar itens do pedido",
                    errorCheck,
                  );
                  return reject({
                    error: true,
                    message: "Erro ao verificar itens do pedido",
                    code: 500,
                  });
                }

                const { total } = resultCheck[0];

                // Se ainda houver itens ativos, NÃO finaliza o pedido
                if (total > 0) {
                  return resolve({
                    success: true,
                    message:
                      "Item cancelado com sucesso. Pedido não foi finalizado pois ainda possui itens ativos.",
                    code: 200,
                  });
                }

                // 2️⃣ Se não houver itens ativos, finaliza o pedido
                const sqlPedido =
                  "UPDATE pedidos SET status_geral = 'Entregue' WHERE id = ?";

                this.#db.query(sqlPedido, [data.id_pedido], (errorPedido) => {
                  if (errorPedido) {
                    console.error(
                      "Erro ao mudar status do pedido",
                      errorPedido,
                    );
                    return reject({
                      error: true,
                      message: "Erro ao mudar status do pedido",
                      code: 500,
                    });
                  }

                  return resolve({
                    success: true,
                    message: "Item cancelado e pedido finalizado com sucesso!",
                    code: 200,
                  });
                });
              },
            );
          },
        );
      } catch (error) {
        console.error("Erro ao mudar item", error);
        return reject({
          error: true,
          message: "Erro interno",
          code: 500,
        });
      }
    });
  }

  getItensPendentesEntrega(id_pedido) {
    return new Promise((resolve, reject) => {
      try {
        const sql = `
    SELECT
    pi.id,
    pi.id_pedido,
    pi.nome_material,
    pi.descricao,
    pi.quantidade,
    pi.preco_acordado,
    DATE_FORMAT(pi.data_entrega_desejada, '%d/%m/%Y') AS data_desejada,
    p.titulo,

    COALESCE(f.nome_empresa, 'Sem fornecedor') AS nome_empresa,
    COALESCE(f.telefone, '---') AS telefone,
    COALESCE(f.email, '---') AS email

FROM pedidos_itens pi
JOIN pedidos p ON p.id = pi.id_pedido
LEFT JOIN pedidos_itens_fornecedores pif ON pif.id_item = pi.id
LEFT JOIN fornecedores f 
       ON f.id = COALESCE(pi.id_fornecedor, pif.id_fornecedor)
WHERE p.status_geral = 'Pendente'
  AND pif.status = 'Entregue Parcial'
  AND pi.id_pedido = ? ;
      `;

        this.#db.query(sql, [id_pedido], (error, results) => {
          if (error) {
            console.error("Erro ao buscar itens pendentes", error);
            return reject({
              error: true,
              message: "Erro ao listar itens pendentes",
              code: 500,
            });
          }

          return resolve(results);
        });
      } catch (error) {
        console.error("Erro ao buscar itens pendentes", error);
        return reject({
          error: true,
          message: "Erro interno",
          code: 500,
        });
      }
    });
  }

  alterarStatusPedido(data) {
    return new Promise(async (resolve, reject) => {
      try {
        // valida se é array
        if (!Array.isArray(data) || data.length === 0) {
          return reject({
            error: true,
            message: "É necessário enviar um array de objetos",
            code: 400,
          });
        }

        const promises = data.map((item) => {
          return new Promise((res, rej) => {
            this.#db.query(
              `UPDATE pedidos_itens_fornecedores SET status = ? WHERE id = ?`,
              [item.status, item.cotacaoId],
              (error) => {
                if (error) {
                  console.error("Erro ao alterar item", error);
                  return rej(error);
                }
                res();
              },
            );
          });
        });

        await Promise.all(promises);

        return resolve({
          success: true,
          message: "Status dos pedidos alterados com sucesso!",
          code: 200,
        });
      } catch (error) {
        console.error("Erro ao alterar status do pedido", error);
        return reject({
          error: true,
          message: "Erro ao alterar status do pedido",
          code: 500,
        });
      }
    });
  }

  deletarCotacao(id) {
    return new Promise((resolve, reject) => {
      try {
        this.#db.query(
          `DELETE FROM pedidos_itens_fornecedores WHERE id = ?`,
          [id],
          (error, response) => {
            if (error) {
              console.error("Erro ao deletar cotação ", error);
              return reject({
                errir: true,
                message: "Erro ao deletar cotação",
                code: 500,
              });
            }
            return resolve({
              success: true,
              message: "Cotação deletada com sucesso!",
              code: 200,
            });
          },
        );
      } catch (error) {
        console.error("Erro ao deletar cotação ", error);
        return reject({
          error: true,
          message: "Erro ao deletar cotação",
          code: 500,
        });
      }
    });
  }

  getPedidosDeEmpresa(id_empresa) {
    return new Promise((resolve, reject) => {
      try {
        this.#db.query(
          `SELECT 
           p.id, 
           p.id_empresa, 
           DATE_FORMAT(p.data_criacao, "%d/%m/%Y") AS data_criacao, 
           p.status_geral,
           p.titulo,
           COUNT(pi.id) AS quantity
         FROM pedidos p
         LEFT JOIN pedidos_itens pi ON pi.id_pedido = p.id
         WHERE p.id_empresa = ?
         GROUP BY p.id`,
          [id_empresa],
          (error, pedidos) => {
            if (error) {
              console.error("Erro ao buscar pedidos ", error);
              return reject({
                error: true,
                message: "Erro ao buscar pedidos",
                code: 500,
              });
            }
            return resolve(pedidos);
          },
        );
      } catch (error) {
        console.error("Erro ao buscar pedidos ", error);
        return reject({
          error: true,
          message: "Erro ao buscar pedidos",
          code: 500,
        });
      }
    });
  }

  getPedidosPendentes() {
    return new Promise((resolve, reject) => {
      try {
        this.#db.query(
          `SELECT 
           p.id, 
           p.id_empresa, 
           DATE_FORMAT(p.data_criacao, "%d/%m/%Y") AS data_criacao, 
           p.status_geral,
           p.titulo,
           COUNT(pi.id) AS quantity
         FROM pedidos p
         LEFT JOIN pedidos_itens pi ON pi.id_pedido = p.id
         WHERE p.status_geral like "Pendente"
         GROUP BY p.id`,
          [],
          (error, pedidos) => {
            if (error) {
              console.error("Erro ao buscar pedidos ", error);
              return reject({
                error: true,
                message: "Erro ao buscar pedidos",
                code: 500,
              });
            }
            return resolve(pedidos);
          },
        );
      } catch (error) {
        console.error("Erro ao buscar pedidos ", error);
        return reject({
          error: true,
          message: "Erro ao buscar pedidos",
          code: 500,
        });
      }
    });
  }

  getCotacao(id_pedido) {
    return new Promise((resolve, reject) => {
      try {
        this.#db.query(
          `
       SELECT 
    pi.id AS id_item,
    pi.nome_material,
    p.status_geral,
    DATE_FORMAT(pi.data_entrega_desejada, "%d/%m/%Y") AS data_entrega_desejada,
    pi.quantidade,
    
    DATE_FORMAT(pi.data_entrega, "%d/%m/%Y") AS data_entrega,
    pi.midia_amostra,

    COALESCE(
        JSON_ARRAYAGG(
            JSON_OBJECT(
                'id', pif.id,
                'id_fornecedor', pif.id_fornecedor,
                'preco', pif.preco,
                'sem_estoque', pif.sem_estoque,
                'forma_pagamento', pif.forma_pagamento_fornecedor,
                'status', pif.status,
                'entrega', pif.entrega
            )
        ),
        JSON_ARRAY()
    ) AS fornecedores,

    CASE
        WHEN JSON_ARRAYAGG(
            JSON_OBJECT(
                'id', pif.id,
                'id_fornecedor', pif.id_fornecedor,
                'preco', pif.preco,
                'forma_pagamento', pif.forma_pagamento_fornecedor,
                'status', pif.status,
                'entrega', pif.entrega
            )
        ) IS NULL OR JSON_ARRAYAGG(
            JSON_OBJECT(
                'id', pif.id,
                'id_fornecedor', pif.id_fornecedor,
                'preco', pif.preco,
                'forma_pagamento', pif.forma_pagamento_fornecedor,
                'status', pif.status,
                'entrega', pif.entrega
            )
        ) = '[]'
        THEN 'Pendente'
        ELSE pi.status_item
    END AS status_item

FROM pedidos p
JOIN pedidos_itens pi 
    ON pi.id_pedido = p.id
LEFT JOIN pedidos_itens_fornecedores pif 
    ON pif.id_item = pi.id
    AND pif.status <> 'Entregue'
   AND pif.status NOT IN ('Aprovado Gestor', 'Aguardando Gestor', 'Não Cotado')

WHERE p.id = ?
AND pi.status_item = 'Pendente'

GROUP BY pi.id;





        `,
          [id_pedido],
          (error, pedidos) => {
            if (error) {
              console.error("Erro ao buscar pedidos ", error);
              return reject({
                error: true,
                message: "Erro ao buscar pedidos",
                code: 500,
              });
            }

            // Parsear fornecedores para array real
            const result = pedidos.map((item) => ({
              ...item,
              fornecedores: JSON.parse(item.fornecedores),
            }));

            return resolve(result);
          },
        );
      } catch (error) {
        console.error("Erro ao buscar pedidos ", error);
        return reject({
          error: true,
          message: "Erro ao buscar pedidos",
          code: 500,
        });
      }
    });
  }

  getCotacaoReprovado(id_pedido) {
    return new Promise((resolve, reject) => {
      try {
        this.#db.query(
          `
       SELECT 
    pi.id AS id_item,
    pi.nome_material,
    p.status_geral,
    DATE_FORMAT(pi.data_entrega_desejada, "%d/%m/%Y") AS data_entrega_desejada,
    pi.quantidade,
    pif.sem_estoque,
    DATE_FORMAT(pi.data_entrega, "%d/%m/%Y") AS data_entrega,
    pi.midia_amostra,

  COALESCE(
    JSON_ARRAYAGG(
        CASE 
            WHEN pif.id IS NOT NULL THEN
                JSON_OBJECT(
                    'id', pif.id,
                    'id_fornecedor', pif.id_fornecedor,
                    'preco', pif.preco,
                    'forma_pagamento', pif.forma_pagamento_fornecedor,
                    'status', pif.status,
                    'entrega', pif.entrega
                )
        END
    ),
    JSON_ARRAY()
) AS fornecedores,

    CASE
        WHEN COUNT(pif.id) = 0 THEN 'Pendente'
        ELSE pi.status_item
    END AS status_item

FROM pedidos p
JOIN pedidos_itens pi 
    ON pi.id_pedido = p.id

LEFT JOIN (
    SELECT 
        pif.*
    FROM pedidos_itens_fornecedores pif
    WHERE pif.status IS NOT NULL
) pif ON pif.id_item = pi.id

WHERE p.id = ?
  AND pi.status_item = 'Pendente'

GROUP BY pi.id;

        `,
          [id_pedido],
          (error, pedidos) => {
            if (error) {
              console.error("Erro ao buscar pedidos ", error);
              return reject({
                error: true,
                message: "Erro ao buscar pedidos",
                code: 500,
              });
            }

            // Parsear fornecedores para array real
            const statusBloqueados = [
              "Aprovado Gestor",
              "Entregue",
              "Entregue Parcial",
            ];

            const result = pedidos
              .map((item) => {
                const fornecedores = JSON.parse(item.fornecedores);

                return {
                  ...item,
                  fornecedores: Array.isArray(fornecedores)
                    ? fornecedores.filter((f) => f && f.id !== null)
                    : [],
                };
              })
              .filter((item) => {
                // se existir ALGUM fornecedor com status bloqueado → remove o item
                return !item.fornecedores.some((f) =>
                  statusBloqueados.includes(f.status),
                );
              });

            return resolve(result);
          },
        );
      } catch (error) {
        console.error("Erro ao buscar pedidos ", error);
        return reject({
          error: true,
          message: "Erro ao buscar pedidos",
          code: 500,
        });
      }
    });
  }

  getCotacoRetornar(id_pedido) {
    return new Promise((resolve, reject) => {
      try {
        this.#db.query(
          `SELECT 
              pi.id AS id_item,
              pi.nome_material,
              p.status_geral,
              DATE_FORMAT(pi.data_entrega_desejada, "%d/%m/%Y") AS data_entrega_desejada,
              pi.quantidade,
              pif.sem_estoque,
              DATE_FORMAT(pi.data_entrega, "%d/%m/%Y") AS data_entrega,
              pi.midia_amostra,

            COALESCE(
              JSON_ARRAYAGG(
                  CASE 
                      WHEN pif.id IS NOT NULL THEN
                          JSON_OBJECT(
                              'id', pif.id,
                              'id_fornecedor', pif.id_fornecedor,
                              'preco', pif.preco,
                              'forma_pagamento', pif.forma_pagamento_fornecedor,
                              'status', pif.status,
                              'entrega', pif.entrega
                          )
                  END
              ),
              JSON_ARRAY()
          ) AS fornecedores,

              CASE
                  WHEN COUNT(pif.id) = 0 THEN 'Pendente'
                  ELSE pi.status_item
              END AS status_item

          FROM pedidos p
          JOIN pedidos_itens pi 
              ON pi.id_pedido = p.id

          LEFT JOIN (
              SELECT 
                  pif.*
              FROM pedidos_itens_fornecedores pif
              WHERE pif.status IS NOT NULL
          ) pif ON pif.id_item = pi.id

          WHERE p.id = ?
            AND pi.status_item = 'Pendente'

          GROUP BY pi.id;`,
          [id_pedido],
          (error, pedidos) => {
            if (error) {
              console.error("Erro ao buscar pedidos ", error);
              return reject({
                error: true,
                message: "Erro ao buscar pedidos",
                code: 500,
              });
            }

            // Parsear fornecedores para array real
            const statusBloqueados = [
              "Aprovado Gestor",
              "Entregue",
              "Entregue Parcial",
              "Reprovado Gestor",
            ];

            const result = pedidos
              .map((item) => {
                const fornecedores = JSON.parse(item.fornecedores);

                return {
                  ...item,
                  fornecedores: Array.isArray(fornecedores)
                    ? fornecedores.filter((f) => f && f.id !== null)
                    : [],
                };
              })
              .filter((item) => {
                // se existir ALGUM fornecedor com status bloqueado → remove o item
                return !item.fornecedores.some((f) =>
                  statusBloqueados.includes(f.status),
                );
              });

            return resolve(result);
          },
        );
      } catch (error) {
        console.error("Erro ao buscar pedidos ", error);
        return reject({
          error: true,
          message: "Erro ao buscar pedidos",
          code: 500,
        });
      }
    });
  }

  getPedidosAprovacao() {
    return new Promise((resolve, reject) => {
      try {
        this.#db.query(
          `SELECT 
    p.id,
    p.id_empresa,
    DATE_FORMAT(p.data_criacao, '%d/%m/%Y') AS data_criacao,
    p.status_geral,
    p.titulo,
    COUNT(DISTINCT pi.id) AS total_itens,
    e.nome
FROM pedidos p
INNER JOIN empresas e 
    ON e.id = p.id_empresa
INNER JOIN pedidos_itens pi 
    ON pi.id_pedido = p.id
INNER JOIN pedidos_itens_fornecedores pif 
    ON pif.id_item = pi.id
   AND pif.status = 'Aguardando Gestor'
WHERE p.status_geral = 'Pendente'
GROUP BY 
    p.id,
    p.id_empresa,
    p.data_criacao,
    p.status_geral,
    p.titulo,
    e.nome;
`,
          [],
          (error, aprovacao) => {
            if (error) {
              console.error(error);
              return reject({
                error: true,
                message: `Não foi possivel ver os pedidos para aprovação ${error}`,
                code: 500,
              });
            }
            return resolve(aprovacao);
          },
        );
      } catch (error) {
        if (error) {
          console.error(error);
          return reject({
            error: true,
            message: `Não foi possivel ver os pedidos para aprovação ${error}`,
            code: 500,
          });
        }
      }
    });
  }

  getPedidosAprovadosGestor(id_fornecedor) {
    return new Promise((resolve, reject) => {
      try {
        this.#db.query(
          `
        SELECT 
    p.id,
    p.id_empresa,
    DATE_FORMAT(p.data_criacao, '%d/%m/%Y') AS data_criacao,
    p.status_geral,
    p.titulo,
    COUNT(DISTINCT pi.id) AS total_itens,
    e.nome
FROM pedidos p
INNER JOIN empresas e 
    ON e.id = p.id_empresa
INNER JOIN pedidos_itens pi 
    ON pi.id_pedido = p.id
INNER JOIN pedidos_itens_fornecedores pif 
    ON pif.id_item = pi.id
   AND pif.status = 'Aprovado Gestor'
   AND pif.id_fornecedor = ?
WHERE p.status_geral = 'Pendente'
GROUP BY 
    p.id,
    p.id_empresa,
    p.data_criacao,
    p.status_geral,
    p.titulo,
    e.nome;

        `,
          [id_fornecedor],
          (error, aprovadoGestor) => {
            if (error) {
              console.error(error);
              return reject({
                error: true,
                message: `Não foi possível ver os pedidos para aprovação`,
                code: 500,
              });
            }
            return resolve(aprovadoGestor);
          },
        );
      } catch (error) {
        console.error(error);
        return reject({
          error: true,
          message: `Não foi possível ver os pedidos para aprovação`,
          code: 500,
        });
      }
    });
  }

  getPedidos() {
    return new Promise((resolve, reject) => {
      try {
        this.#db.query(
          `
          SELECT 
    p.id,
    p.id_empresa,
    DATE_FORMAT(p.data_criacao, '%d/%m/%Y') AS data_criacao,
    p.status_geral,
    p.titulo,

    COUNT(DISTINCT pi.id) AS total_itens,

    COUNT(DISTINCT CASE 
        WHEN pif.status = 'Cotado' THEN pi.id
    END) AS total_cotados,

    e.nome,
    /* flags de status nos itens/fornecedores */
    MAX(CASE WHEN pif.status = 'Reprovado Gestor' THEN 1 ELSE 0 END) AS tem_reprovado_gestor,
    MAX(CASE WHEN pif.status = 'Aguardando Gestor' THEN 1 ELSE 0 END) AS tem_aguardando_gestor,
    MAX(CASE WHEN pif.status = 'Cotado' THEN 1 ELSE 0 END) AS tem_cotado

FROM pedidos p
INNER JOIN empresas e 
    ON e.id = p.id_empresa
INNER JOIN pedidos_itens pi 
    ON pi.id_pedido = p.id
LEFT JOIN pedidos_itens_fornecedores pif 
    ON pif.id_item = pi.id

WHERE p.status_geral = 'Pendente'

  AND (
       pi.status_item = 'Pendente'
       OR pif.status IN ('Pendente', 'Cotado', 'Reprovado Gestor', 'Recusado Gestor')
  )

 

  -- ❌ Nunca mostrar se existir Aprovado Gestor
  AND NOT EXISTS (
      SELECT 1
      FROM pedidos_itens_fornecedores pif_ap
      WHERE pif_ap.id_item = pi.id
        AND pif_ap.status = 'Aprovado Gestor'
  )

    AND NOT EXISTS (
      SELECT 1
      FROM pedidos_itens_fornecedores pif_et
      WHERE pif_et.id_item = pi.id
        AND pif_et.status = 'Em_transito'
  )
    AND NOT EXISTS (
      SELECT 1
      FROM pedidos_itens_fornecedores pif_en
      WHERE pif_en.id_item = pi.id
        AND pif_en.status = 'Entregue Parcial'
  )
    AND NOT EXISTS (
      SELECT 1
      FROM pedidos_itens_fornecedores pif_e
      WHERE pif_e.id_item = pi.id
        AND pif_e.status = 'Entregue'
  )

GROUP BY 
    p.id,
    p.id_empresa,
    p.data_criacao,
    p.status_geral,
    p.titulo,
    e.nome;




          `,
          [],
          (error, pedidos) => {
            if (error) {
              console.error("Erro ao buscar pedidos ", error);
              return reject({
                error: true,
                message: "Erro ao buscar pedidos",
                code: 500,
              });
            }
            return resolve(pedidos);
          },
        );
      } catch (error) {
        console.error("Erro ao buscar pedidos ", error);
        return reject({
          error: true,
          message: "Erro ao buscar pedidos",
          code: 500,
        });
      }
    });
  }

  getItensPedido(id_pedido) {
    return new Promise((resolve, reject) => {
      try {
        this.#db.query(
          `SELECT * FROM pedidos_itens WHERE id_pedido = ?`,
          [id_pedido],
          (error, results) => {
            console.log(results);
            if (error) {
              console.error("Erro ao buscar pedidos ", error);
              return reject({
                error: true,
                message: "Erro ao buscar pedidos",
                code: 500,
              });
            }
            return resolve(results);
          },
        );
      } catch (error) {
        console.error("Erro ao buscar pedidos ", error);
        return reject({
          error: true,
          message: "Erro ao buscar pedidos",
          code: 500,
        });
      }
    });
  }

  getUmPedido(id) {
    return new Promise(async (resolve, reject) => {
      try {
        this.#db.query(
          `SELECT 
            id AS id_pedido, 
            DATE_FORMAT(data_criacao, "%d/%m/%Y") AS data_criado, 
            status_geral
           FROM pedidos
           WHERE id = ?`,
          [id],
          (error, results) => {
            if (error) {
              console.error("Erro ao buscar pedido ", error);
              return reject({
                error: true,
                message: "Erro ao buscar pedido",
                code: 500,
              });
            }

            if (results.length === 0) {
              return reject({
                error: true,
                message: "Pedido não encontrado",
                code: 404,
              });
            }

            return resolve(results[0]);
          },
        );
      } catch (error) {
        console.error("Erro ao buscar pedido ", error);
        return reject({
          error: true,
          message: "Erro ao buscar pedido",
          code: 500,
        });
      }
    });
  }

  getPedidosDeFornecedor(id_fornecedor) {
    return new Promise(async (resolve, reject) => {
      try {
        this.#db.query(
          `SELECT 
              p.id AS id_pedido, 
              DATE_FORMAT(p.data_criacao, '%d/%m/%Y') AS data_criado, 
              p.status_geral,
              p.titulo,
              COUNT(pi.id) AS quantity
          FROM pedidos AS p
          JOIN empresas e 
              ON e.id = p.id_empresa
          JOIN pedidos_itens AS pi 
              ON pi.id_pedido = p.id
          JOIN pedidos_itens_fornecedores AS pif 
              ON pif.id_item = pi.id   
          WHERE pif.id_fornecedor = ? AND pif.status LIKE "Aguardando cotação" 
          GROUP BY 
              p.id, 
              p.data_criacao,
              p.status_geral`,
          [id_fornecedor],
          (error, results) => {
            if (error) {
              console.error("Erro ao buscar pedido ", error);
              return reject({
                error: true,
                message: "Erro ao buscar pedido",
                code: 500,
              });
            }

            if (results.length === 0) {
              return reject({
                error: true,
                message: "nenhum não encontrado",
                code: 400,
              });
            }

            return resolve(results);
          },
        );
      } catch (error) {
        console.error("Erro ao buscar pedido ", error);
        return reject({
          error: true,
          message: "Erro ao buscar pedido",
          code: 500,
        });
      }
    });
  }

  getPedidosDeFornecedorAprovado(id_fornecedor) {
    return new Promise(async (resolve, reject) => {
      try {
        this.#db.query(
          `SELECT 
              p.id AS id_pedido, 
              DATE_FORMAT(p.data_criacao, '%d/%m/%Y') AS data_criado, 
              p.status_geral,
              p.titulo,
              COUNT(pi.id) AS quantity
          FROM pedidos AS p
          JOIN pedidos_itens AS pi
              ON pi.id_pedido = p.id
          JOIN pedidos_itens_fornecedores AS pif
              ON pif.id_item = pi.id
          WHERE pif.id_fornecedor = ? AND p.status_geral LIKE "Pendente" AND pif.status ="Aprovado Gestor"
          GROUP BY 
              p.id, 
              p.data_criacao,
              p.status_geral`,
          [id_fornecedor],
          (error, results) => {
            if (error) {
              console.error("Erro ao buscar pedido ", error);
              return reject({
                error: true,
                message: "Erro ao buscar pedido",
                code: 500,
              });
            }

            if (results.length === 0) {
              return reject({
                error: true,
                message: "nenhum pedido encontrado",
                code: 400,
              });
            }

            return resolve(results);
          },
        );
      } catch (error) {
        console.error("Erro ao buscar pedido ", error);
        return reject({
          error: true,
          message: "Erro ao buscar pedido",
          code: 500,
        });
      }
    });
  }

  getPedidosGestor(id_pedido) {
    return new Promise(async (resolve, reject) => {
      try {
        this.#db.query(
          `SELECT
              pi.id AS id_item,
              pi.nome_material,
              pi.quantidade,
              pi.descricao,
              fo.nome_empresa AS nome,
              p.id AS id_pedido,
              DATE_FORMAT(pif.entrega, "%d/%m/%Y") AS data_entrega,
              pif.preco AS preco_acordado,
              pif.status AS status_cotacao,
              pif.id AS id_cotacao,
              pif.sem_estoque AS sem_estoque,
              pi.midia_amostra,
              p.status_geral,
              DATE_FORMAT(pi.data_entrega_desejada, "%d/%m/%Y") AS data_entrega_desejada,
              pi.quantidade
            FROM pedidos_itens pi, fornecedores fo, pedidos p, pedidos_itens_fornecedores pif
            WHERE pi.id_pedido = ? AND
            pif.id_fornecedor = fo.id AND
            pif.status = "Aguardando Gestor" AND
            pi.id_pedido = p.id AND
            pi.id = pif.id_item
            `,
          [id_pedido],
          (error, results) => {
            if (error) {
              console.error("Erro ao buscar itens do pedido ", error);
              return reject({
                error: true,
                message: "Erro ao buscar itens do pedido",
                code: 500,
              });
            }
            return resolve(results);
          },
        );
      } catch (error) {
        console.error("Erro ao buscar pedido ", error);
        return reject({
          error: true,
          message: "Erro ao buscar pedido",
          code: 500,
        });
      }
    });
  }

  /**
   * Busca um único pedido
   */
  getUmPedidoComItens(id_pedido) {
    return new Promise(async (resolve, reject) => {
      try {
        const pedido = await this.getUmPedido(id_pedido);

        const itens = await new Promise((resolve, reject) => {
          this.#db.query(
            `SELECT
              pi.id AS id_item,
              pi.id_pedido,
              pi.nome_material,
              pi.descricao,
              pi.midia_amostra,
              DATE_FORMAT(pi.data_entrega_desejada, "%d/%m/%Y") AS data_entrega_desejada,
              pi.quantidade
            FROM pedidos_itens pi
            WHERE pi.id_pedido = ?`,
            [id_pedido],
            (error, results) => {
              if (error) {
                console.error("Erro ao buscar itens do pedido ", error);
                return reject({
                  error: true,
                  message: "Erro ao buscar itens do pedido",
                  code: 500,
                });
              }
              resolve(results);
            },
          );
        });

        const retorno = {
          id_pedido: pedido.id_pedido,
          data_criado: pedido.data_criado,
          status_geral: pedido.status_geral,
          itens: itens,
        };

        return resolve(retorno);
      } catch (error) {
        console.error("Erro ao buscar pedido ", error);
        return reject({
          error: true,
          message: "Erro ao buscar pedido",
          code: 500,
        });
      }
    });
  }

  getUmPedidoComItensDeForncedor(id_pedido, id_fornecedor) {
    return new Promise(async (resolve, reject) => {
      try {
        const itens = await new Promise((resolve, reject) => {
          this.#db.query(
            `SELECT
              pi.id AS id_item,
              pi.id_pedido,
              pi.nome_material,
              pi.descricao,
              pi.midia_amostra,
              DATE_FORMAT(pi.data_entrega_desejada, "%d/%m/%Y") AS data_entrega_desejada,
              pi.quantidade
            FROM pedidos_itens pi
            JOIN pedidos_itens_fornecedores pif 
                ON pif.id_item = pi.id
            JOIN fornecedores f
                ON f.id = pif.id_fornecedor
            WHERE 
                pi.id_pedido = ?
                AND pif.id_fornecedor = ?
                AND (pif.status LIKE "Aguardando cotação" OR pif.status LIKE "Reprovado");
            `,
            [id_pedido, id_fornecedor],
            (error, results) => {
              if (error) {
                console.error("Erro ao buscar itens do pedido ", error);
                return reject({
                  error: true,
                  message: "Erro ao buscar itens do pedido",
                  code: 500,
                });
              }
              resolve(results);
            },
          );
        });

        return resolve(itens);
      } catch (error) {
        console.error("Erro ao buscar pedido ", error);
        return reject({
          error: true,
          message: "Erro ao buscar pedido",
          code: 500,
        });
      }
    });
  }

  getUmPedidoComItensDeForncedorAprovado(id_pedido, id_fornecedor) {
    return new Promise(async (resolve, reject) => {
      try {
        const itens = await new Promise((resolve, reject) => {
          this.#db.query(
            `SELECT
              pi.id AS id_item,
              pi.id_pedido,
              pi.nome_material,
              pi.descricao,
              pi.midia_amostra,
              DATE_FORMAT(pi.data_entrega_desejada, "%d/%m/%Y") AS data_entrega_desejada,
              pi.quantidade,
              p.titulo,
              COALESCE(pi.preco_acordado, pif.preco) AS preco_acordado,
              DATE_FORMAT(pif.entrega, "%d/%m/%Y") AS data_entrega,
              pi.data_entrega_desejada
            FROM pedidos_itens pi, pedidos_itens_fornecedores pif, fornecedores, pedidos p
            WHERE pi.id_pedido = ? AND
            p.id = pi.id_pedido AND
            pif.id_item = pi.id AND
            pif.id_fornecedor = ? AND
            pif.id_fornecedor = fornecedores.id AND
            pif.status = "Aprovado Gestor"
            `,
            [id_pedido, id_fornecedor],
            (error, results) => {
              if (error) {
                console.log(results);
                console.error("Erro ao buscar itens do pedido ", error);
                return reject({
                  error: true,
                  message: "Erro ao buscar itens do pedido",
                  code: 500,
                });
              }
              resolve(results);
            },
          );
        });

        return resolve(itens);
      } catch (error) {
        console.error("Erro ao buscar pedido ", error);
        return reject({
          error: true,
          message: "Erro ao buscar pedido",
          code: 500,
        });
      }
    });
  }

  getPedidosAprovados() {
    return new Promise((resolve, reject) => {
      try {
        this.#db.query(
          `SELECT
    p.id,
    DATE_FORMAT(p.data_criacao, '%d/%m/%Y') AS data_criado,
    p.status_geral,
    p.id_empresa,
    p.titulo,
    e.nome AS nome_empresa,

    /* flags de status nos itens/fornecedores */
    MAX(CASE WHEN pif.status = 'Reprovado Gestor' THEN 1 ELSE 0 END) AS tem_reprovado_gestor,
    MAX(CASE WHEN pif.status = 'Aguardando Gestor' THEN 1 ELSE 0 END) AS tem_aguardando_gestor,
    MAX(CASE WHEN pif.status = 'Aprovado Gestor' THEN 1 ELSE 0 END) AS tem_aprovado_gestor,
    MAX(CASE WHEN pif.status = 'Em_transito' THEN 1 ELSE 0 END) AS tem_em_transito,
    MAX(CASE WHEN pif.status = 'Entregue' THEN 1 ELSE 0 END) AS tem_entregue,
    MAX(CASE WHEN pif.status = 'Cotado' THEN 1 ELSE 0 END) AS tem_cotado

FROM pedidos AS p
INNER JOIN empresas AS e 
    ON e.id = p.id_empresa
INNER JOIN pedidos_itens AS pi 
    ON pi.id_pedido = p.id
INNER JOIN pedidos_itens_fornecedores AS pif 
    ON pif.id_item = pi.id

WHERE 
    p.status_geral IN ('Aprovado', 'Entregue', 'Pendente')
AND
    pif.status IN (
        'Aprovado Gestor',
        'Em_transito',
        'Entregue',
        'Não entregue',
        'Cancelado'
    )

GROUP BY 
    p.id,
    p.data_criacao,
    p.status_geral,
    p.id_empresa,
    p.titulo,
    e.nome;

        `,
          (error, result) => {
            if (error) {
              console.error("Erro ao buscar pedidos aprovados ", error);
              return reject({
                error: true,
                message: "Erro ao buscar pedidos aprovados",
                code: 500,
              });
            }
            resolve(result);
          },
        );
      } catch (error) {
        console.error("Erro ao buscar pedidos aprovados ", error);
        return reject({
          error: true,
          message: "Erro ao buscar pedidos aprovados",
          code: 500,
        });
      }
    });
  }

  getItensAprovados(id_pedido) {
    return new Promise((resolve, reject) => {
      this.#db.query(
        `SELECT 
        pi.id,
        pi.id_pedido,
        pi.nome_material,
        pi.descricao,
        p.titulo,
        pi.midia_amostra,
        pif.preco AS preco_acordado,
        pif.entrega AS data_entrega_confirmada,
        pif.status AS status_item,
        pif.forma_pagamento_fornecedor,
        pi.quantidade,
        pif.sem_estoque,
        f.id AS id_fornecedor,
        f.nome_empresa AS nome_fornecedor
      FROM pedidos_itens pi
      LEFT JOIN pedidos_itens_fornecedores pif ON pif.id_item = pi.id
      LEFT JOIN pedidos p ON p.id = pi.id_pedido
      LEFT JOIN fornecedores f ON f.id = pif.id_fornecedor
      WHERE pif.id_pedido = ?
      AND pif.status IN (
        'Aprovado Gestor',
        'Aguardando Gestor',
        'Aprovado',
        'Em_transito',
        'Entregue',
        'Não entregue',
        'Cancelado'
      )`,
        [id_pedido],
        (error, results) => {
          if (error) {
            return reject({
              error: true,
              message: "Erro ao buscar itens finalizados",
              code: 500,
            });
          }

          if (!results.length) {
            return resolve({ itens: [], imagens: [] });
          }

          const idPedido = results[0].id_pedido;

          this.#db.query(
            `SELECT
            nota_fiscal,
            id_fornecedor,
            foto_entregador,
            assinatura
           FROM pedidos_recebimentos
           WHERE id_pedido = ?`,
            [idPedido],
            (error2, results2) => {
              if (error2) {
                return reject({
                  error: true,
                  message: "Erro ao buscar as imagens do recebimento",
                  code: 500,
                });
              }

              resolve({
                itens: results,
                imagens: results2,
              });
            },
          );
        },
      );
    });
  }

  responderCotacao(data) {
    return new Promise(async (resolve, reject) => {
      try {
        for (const item of data.itens) {
          let dataFormatada = null;
          if (item.data_entrega) {
            const partes = item.data_entrega.split("/");
            if (partes.length === 3) {
              const [dia, mes, ano] = partes;
              dataFormatada = `${ano}-${mes}-${dia}`; // Resultado: "2025-12-10"
            }
          }

          await new Promise((res, rej) => {
            this.#db.query(
              `UPDATE pedidos_itens_fornecedores 
                   SET preco = ?,
                       entrega = ?,
                       sem_estoque = ?,
                       forma_pagamento_fornecedor = ?,
                       status = 'Cotado'
                 WHERE id_fornecedor = ? AND id_item = ? AND status = 'Aguardando cotação'`,
              [
                item.preco,
                dataFormatada,
                item.sem_estoque ? 1 : 0,
                data.condicao_pagamento,
                data.id_fornecedor,
                item.id_item,
              ],
              (error) => {
                if (error) {
                  console.error("Erro ao atualizar cotacao ", error);
                  return rej({
                    error: true,
                    message: `Erro ao atualizar cotação`,
                    code: 500,
                  });
                }
                res();
              },
            );
          });
        }

        const fornecedor = await this.#fornecedores.getUmFornecedor(
          data.id_fornecedor,
        );
        const menssagem = ``;

        const compradores = await this.#compradores.getCompradores();

        // // for (const comprador of compradores) {
        //     await sendMessage();
        // // }
        return resolve({
          success: true,
          message: `Cotação aceita com sucesso`,
          code: 200,
        });
      } catch (error) {
        console.error("Erro ao responder cotação ", error);
        return reject({
          error: true,
          message: "Erro ao responder cotação",
          code: 500,
        });
      }
    });
  }

  /**
   * Deleta um pedido
   */
  delete(id) {
    return new Promise((resolve, reject) => {
      this.#db.query(
        `DELETE FROM pedidos WHERE id = ?`,
        [id],
        (error, response) => {
          if (error) {
            console.error("Erro ao deletar pedido ", error);
            return reject({
              error: true,
              message: "Erro ao deletar pedido",
              code: 500,
            });
          }

          if (response.affectedRows === 0) {
            return reject({
              error: true,
              message: "Pedido não encontrado",
              code: 404,
            });
          }

          return resolve({
            success: true,
            message: "Pedido deletado com sucesso!",
            code: 200,
          });
        },
      );
    });
  }

  //   salvarDecisoesPedido(data) {
  //   return new Promise(async (resolve, reject) => {
  //     try {
  //       // 1) Atualiza cada item individualmente
  //       for (const item of data.decisoes) {
  //         await new Promise((res, rej) => {
  //           this.#db.query(
  //             `UPDATE pedidos_itens
  //                SET status_item = ?
  //              WHERE id = ? AND id_pedido = ?`,
  //             [item.status, item.id_item, data.id_pedido],
  //             (error) => {
  //               if (error) {
  //                 console.error("Erro ao salvar decisão ", error);
  //                 return rej({ error: true, message: "Erro ao salvar decisão", code: 500 });
  //               }
  //               res();
  //             }
  //           );
  //         });
  //       }

  //       // 2) Consulta o status geral dos itens do pedido
  //       const itens = await new Promise((res, rej) => {
  //         this.#db.query(
  //           `SELECT status_item FROM pedidos_itens WHERE id_pedido = ?`,
  //           [data.id_pedido],
  //           (error, results) => {
  //             if (error) {
  //               console.error("Erro ao buscar itens ", error);
  //               return rej({ error: true, message: "Erro ao consultar itens", code: 500 });
  //             }
  //             res(results);
  //           }
  //         );
  //       });

  //       const temReprovado = itens.some(i => i.status_item === "Recusado");
  //       const todosAprovados = itens.every(i => i.status_item === "Aprovado");

  //       let novoStatusGeral = "";

  //       if (temReprovado) novoStatusGeral = "Cancelado";
  //       else if (todosAprovados) novoStatusGeral = "Aprovado";

  //       // 3) Atualiza o pedido com o status geral calculado
  //       await new Promise((res, rej) => {
  //         this.#db.query(
  //           `UPDATE pedidos
  //               SET status_geral = ?
  //             WHERE id = ?`,
  //           [novoStatusGeral, data.id_pedido],
  //           (error) => {
  //             if (error) {
  //               console.error("Erro ao atualizar status geral", error);
  //               return rej({ error: true, message: "Erro ao atualizar status geral", code: 500 });
  //             }
  //             res();
  //           }
  //         );
  //       });

  //       return resolve({ success: true, message: "Decisões salvas e status geral atualizado", code: 200 });

  //     } catch (err) {
  //       console.error("Erro ao processar decisões ", err);
  //       return reject({ error: true, message: "Erro interno", code: 500 });
  //     }
  //   });
  // }

  // salvarDecisoesPedido(data) {
  //   return new Promise(async (resolve, reject) => {
  //     try {
  //       for (const item of data.decisoes) {

  //         let updateQuery = "";
  //         let params = [];

  //         if (item.status === "Reprovado") {
  //           updateQuery = `
  //           UPDATE pedidos_itens_fornecedores
  //              SET status = ?,
  //            WHERE id = ?`;
  //           params = [item.status, item.id_cotacao];

  //         } else {
  //           updateQuery = `
  //           UPDATE pedidos_itens
  //              SET status_item = ?
  //            WHERE id = ? AND id_pedido = ?`;
  //           params = [item.status, item.id_cotacao];
  //         }

  //         await new Promise((res, rej) => {
  //           this.#db.query(updateQuery, params, (error) => {
  //             if (error) {
  //               console.error("Erro ao salvar decisão ", error);
  //               return rej({
  //                 error: true,
  //                 message: "Erro ao salvar decisão",
  //                 code: 500
  //               });
  //             }
  //             res();
  //           });
  //         });
  //       }

  //       const itens = await new Promise((res, rej) => {
  //         this.#db.query(
  //           `SELECT status_item FROM pedidos_itens WHERE id_pedido = ?`,
  //           [data.id_pedido],
  //           (error, results) => {
  //             if (error) {
  //               console.error("Erro ao buscar itens ", error);
  //               return rej({
  //                 error: true,
  //                 message: "Erro ao consultar itens",
  //                 code: 500
  //               });
  //             }
  //             res(results);
  //           }
  //         );
  //       });

  //       const temReprovado = itens.some(i => i.status_item === "Reprovado");
  //       const todosAprovados = itens.every(i => i.status_item === "Aprovado");

  //       let novoStatusGeral = "Em análise";

  //       if (temReprovado) novoStatusGeral = "Recusado";
  //       else if (todosAprovados) novoStatusGeral = "Aprovado";

  //       await new Promise((res, rej) => {
  //         this.#db.query(
  //           `UPDATE pedidos
  //             SET status_geral = ?
  //           WHERE id = ?`,
  //           [novoStatusGeral, data.id_pedido],
  //           (error) => {
  //             if (error) {
  //               console.error("Erro ao atualizar status geral", error);
  //               return rej({
  //                 error: true,
  //                 message: "Erro ao atualizar status geral",
  //                 code: 500
  //               });
  //             }
  //             res();
  //           }
  //         );
  //       });

  //       return resolve({
  //         success: true,
  //         message: "Decisões salvas e status geral atualizado",
  //         code: 200
  //       });

  //     } catch (err) {
  //       console.error("Erro ao processar decisões ", err);
  //       return reject({ error: true, message: "Erro interno", code: 500 });
  //     }
  //   });
  // }

  /* salvarDecisoesPedido(data) {
  return new Promise((resolve, reject) => {
    try {
      this.#db.query(
        `UPDATE pedidos_itens_fornecedores SET status = ? WHERE id = ?`,
        [data.status, data.id_cotacao],
        (error, results) => {
          if (error) {
            console.error("Erro ao salvar decisão: ", error);
            return reject({ error: true, message: "Erro ao salvar decisão", code: 500 });
          }

          
          this.#db.query(
            `SELECT id_fornecedor FROM pedidos_itens_fornecedores WHERE id = ?`,
            [data.id_cotacao],
            (error, resultsIdFornecedor) => {
              if (error) {
                console.error("Erro ao buscar id do fornecedor", error);
                return reject({ error: true, message: "Erro ao buscar id do fornecedor", code: 500 });
              }

              if (resultsIdFornecedor.length === 0) {
                 return resolve({ error: false, message: "Status atualizado.", code: 200 });
              }

              const idFornecedor = resultsIdFornecedor[0].id_fornecedor;

              
              this.#db.query(
                `SELECT telefone, email, nome_empresa FROM fornecedores WHERE id = ?`,
                [idFornecedor], 
                 async(error, resultTelefoneFornecedor) => {
                  if (error) {
                    console.error("Erro ao buscar telefone");
                    return reject({ error: true, message: "Erro ao buscar telefone", code: 500 });
                  }

                  if (resultTelefoneFornecedor.length > 0) {
                    const telefone = resultTelefoneFornecedor[0].telefone;
                    const email = resultTelefoneFornecedor[0].email;
                    const nome_empresa = resultTelefoneFornecedor[0].nome_empresa;
                    
                    
                    let mensagemTexto = "";

                    if (data.status === 'Reprovado') {
                        mensagemTexto = "Olá caro fornecedor, sua cotação foi recusada.";
                    } else {
                        await enviarEmail({ fornecedor: nome_empresa, id_cotacao: data.id_cotacao, email: email });
                        mensagemTexto = "Olá caro fornecedor, sua cotação foi aprovada!";
                        await sendMessage(mensagemTexto, telefone);
                    }

                    
                    
                  }

                  // Finaliza a Promise com sucesso
                  return resolve({
                    error: false,
                    message: "Decisão salva e fornecedor notificado",
                    code: 200
                  });
                }
              );
            }
          );
        }
      );
    } catch (err) {
      console.error("Erro ao processar decisões: ", err);
      return reject({ error: true, message: "Erro interno", code: 500 });
    }
  });
} */

  async salvarDecisoesPedido(data) {
    try {
      const updatePromises = data.decisoes.map((item) => {
        return new Promise((resolve, reject) => {
          // 1️⃣ Atualiza status
          this.#db.query(
            `UPDATE pedidos_itens_fornecedores
           SET status = ?
           WHERE id = ?`,
            [item.status, item.id_cotacao],
            (error) => {
              if (error) {
                console.error(`Erro no item ${item.id_cotacao}:`, error);
                return reject(error);
              }

              // 2️⃣ Se aprovado, continua fluxo
              if (item.status === "Aprovado Gestor") {
                this.#db.query(
                  `SELECT id_fornecedor
                 FROM pedidos_itens_fornecedores
                 WHERE id = ?`,
                  [item.id_cotacao],
                  async (error, results) => {
                    if (error) {
                      console.error("Erro ao buscar fornecedor:", error);
                      return reject(error);
                    }

                    if (!results.length) {
                      return resolve();
                    }

                    try {
                      const idFornecedor = results[0].id_fornecedor;

                      const fornecedor =
                        await this.#fornecedores.getUmFornecedor(idFornecedor);

                      if (!fornecedor?.telefone) {
                        return resolve();
                      }

                      await sendMessage(
                        `Sua cotação foi aprovada pelo gestor!

Acesse o sistema para dar andamento ao pedido dos itens aprovados: https://obracroissantecia.com.br

Em caso de dúvidas, entre em contato com (11) 97259-7464. Mensagem automatica, nao responder.`,
                        fornecedor.telefone,
                      );

                      resolve();
                    } catch (err) {
                      reject(err);
                    }
                  },
                );
              } else {
                // Se não for aprovado, resolve direto
                resolve();
              }
            },
          );
        });
      });

      await Promise.all(updatePromises);

      return { success: true };
    } catch (error) {
      console.error("Erro ao processar decisões:", error);
      throw { error: true, message: "Erro ao salvar itens", code: 500 };
    }
  }

  async voltarCotado(id_pedido) {
    return new Promise((resolve, reject) => {
      this.#db.query(
        `UPDATE pedidos_itens_fornecedores SET status = 'Cotado' WHERE id_pedido = ? AND status = 'Aguardando Gestor'`,
        [id_pedido],
        (error, response) => {
          if (error) {
            console.error("Erro ao voltar cotação:", error);
            return reject(error);
          }
          return resolve({
            success: true,
            message: "Pedido retornado para cotação com sucesso!",
            code: 200,
          });
        },
      );
    });
  }

  /**
   * Atualiza um pedido
   */
  update(data) {
    return new Promise((resolve, reject) => {
      try {
        const fields = [];
        const values = [];

        if (data.id_empresa) {
          fields.push("id_empresa = ?");
          values.push(data.id_empresa);
        }
        if (data.data_criacao) {
          fields.push("data_criacao = ?");
          values.push(data.data_criacao);
        }
        if (data.status_geral) {
          fields.push("status_geral = ?");
          values.push(data.status_geral);
        }

        if (fields.length === 0) {
          return reject({
            error: true,
            message: "Nenhum dado para atualizar",
            code: 422,
          });
        }

        values.push(data.id);

        const query = `
          UPDATE pedidos
          SET ${fields.join(", ")}
          WHERE id = ?
        `;

        this.#db.query(query, values, (error, response) => {
          if (error) {
            console.error("Erro ao atualizar pedido ", error);
            return reject({
              error: true,
              message: "Erro ao atualizar pedido",
              code: 500,
            });
          }

          if (response.affectedRows === 0) {
            return reject({
              error: true,
              message: "Pedido não encontrado",
              code: 404,
            });
          }

          return resolve({
            success: true,
            message: "Pedido atualizado com sucesso!",
            code: 200,
          });
        });
      } catch (error) {
        console.error("Erro ao atualizar pedido ", error);
        return reject({
          error: true,
          message: "Erro ao atualizar pedido",
          code: 500,
        });
      }
    });
  }
}

module.exports = PedidosRepository;
