const Pedidos = require("./pedidos.repository");

class PedidosController {
  #pedidos;

  constructor(pedidosRepository = new Pedidos()) {
    this.#pedidos = pedidosRepository;
  }

  /**
   * Cadastra um operário
   */
  register() {
    return async (req, res) => {
      try {
        const { trueId: empresaId } = req.user;
        const data = req.body;
        const files = req.files || [];

        data.id_empresa = empresaId;

        if (typeof data.itens === "string") {
          data.itens = JSON.parse(data.itens);
        }

        const response = await this.#pedidos.register(data, files);

        return res.status(response.code).json(response);
      } catch (error) {
        console.error(error);
        return res.status(500).json(error);
      }
    };
  }

  voltarCotado() {
    return async (req, res) => {
      try {
        const id_pedido = req.params.id_pedido;
        const response = await this.#pedidos.voltarCotado(id_pedido);
        if (response.error) return res.status(response.code).json(response);
        return res.status(response.code).json(response);
      } catch (error) {
        console.error(error);
        return res.status(error.code || 500).json(error);
      }
    };
  }

  updateItens() {
    return async (req, res) => {
      try {
        const id = req.params.id;
        const data = req.body;

        const response = await this.#pedidos.updateItens(data, id);
        return res.status(response.code).json(response);
      } catch (error) {
        console.error(error);
        return res.status(error.code || 500).json(error);
      }
    };
  }

  getPedidosAprovacao() {
    return async (req, res) => {
      try {
        const response = await this.#pedidos.getPedidosAprovacao();
        if (response.error) return res.status(response.code).json(response);
        return res.status(200).json(response);
      } catch (error) {
        console.error(error);
        return res.status(error.code || 500).json(error);
      }
    };
  }

  getPedidosAprovadosGestor() {
    return async (req, res) => {
      try {
        const id_fornecedor = req.params.id_fornecedor;
        const response =
          await this.#pedidos.getPedidosAprovadosGestor(id_fornecedor);
        if (response.error) return res.status(response.code).json(response);
        return res.status(200).json(response);
      } catch (error) {
        console.error(error);
        return res.status(error.code || 500).json(error);
      }
    };
  }

  getPedidos() {
    return async (req, res) => {
      try {
        const response = await this.#pedidos.getPedidos();

        if (response.error) return res.status(response.code).json(response);
        return res.status(200).json(response);
      } catch (error) {
        console.error(error);
        return res.status(error.code || 500).json(error);
      }
    };
  }

  getItensPedido() {
    return async (req, res) => {
      try {
        const id_pedido = req.params.id_pedido;
        const response = await this.#pedidos.getItensPedido(id_pedido);
        if (response.error) return res.status(response.code).json(response);
        return res.status(200).json(response);
      } catch (error) {
        console.error(error);
        return res.status(error.code || 500).json(error);
      }
    };
  }

  getPedidosPendentes() {
    return async (req, res) => {
      try {
        const response = await this.#pedidos.getPedidosPendentes();

        if (response.error) return res.status(response.code).json(response);
        return res.status(200).json(response);
      } catch (error) {
        console.error(error);
        return res.status(error.code || 500).json(error);
      }
    };
  }

  getCotacao() {
    return async (req, res) => {
      try {
        const response = await this.#pedidos.getCotacao(req.params.id_pedido);

        if (response.error) return res.status(response.code).json(response);
        return res.status(200).json(response);
      } catch (error) {
        console.error(error);
        return res.status(error.code || 500).json(error);
      }
    };
  }

  getCotacaoReprovado() {
    return async (req, res) => {
      try {
        const response = await this.#pedidos.getCotacaoReprovado(
          req.params.id_pedido,
        );

        if (response.error) return res.status(response.code).json(response);
        return res.status(200).json(response);
      } catch (error) {
        console.error(error);
        return res.status(error.code || 500).json(error);
      }
    };
  }

  getCotacoRetornar() {
    return async (req, res) => {
      try {
        const response = await this.#pedidos.getCotacoRetornar(
          req.params.id_pedido,
        );
        if (response.error) return res.status(response.code).json(response);
        return res.status(200).json(response);
      } catch (error) {
        console.error(error);
        return res.status(error.code || 500).json(error);
      }
    };
  }

  getPedidosDeEmpresa() {
    return async (req, res) => {
      try {
        const response = await this.#pedidos.getPedidosDeEmpresa(
          req.params.id_empresa,
        );

        if (response.error) return res.status(response.code).json(response);
        return res.status(200).json(response);
      } catch (error) {
        console.error(error);
        return res.status(error.code || 500).json(error);
      }
    };
  }

  /**
   * Busca um único operário
   */
  getUmPedido() {
    return async (req, res) => {
      try {
        const id = req.params.id;

        const response = await this.#pedidos.getUmPedido(id);

        if (response.error) return res.status(response.code).json(response);
        return res.status(200).json(response);
      } catch (error) {
        console.error(error);
        return res.status(error.code || 500).json(error);
      }
    };
  }

  getUmPedidoComItens() {
    return async (req, res) => {
      try {
        const id_pedido = req.params.id_pedido;

        const response = await this.#pedidos.getUmPedidoComItens(id_pedido);

        if (response.error) return res.status(response.code).json(response);
        return res.status(200).json(response);
      } catch (error) {
        console.error(error);
        return res.status(error.code || 500).json(error);
      }
    };
  }

  getUmPedidoComItensDeForncedor() {
    return async (req, res) => {
      try {
        const id_pedido = req.params.id_pedido;
        const id_fornecedor = req.params.id_fornecedor;

        const response = await this.#pedidos.getUmPedidoComItensDeForncedor(
          id_pedido,
          id_fornecedor,
        );

        if (response.error) return res.status(response.code).json(response);
        return res.status(200).json(response);
      } catch (error) {
        console.error(error);
        return res.status(error.code || 500).json(error);
      }
    };
  }

  getUmPedidoComItensDeForncedorAprovado() {
    return async (req, res) => {
      try {
        const id_pedido = req.params.id_pedido;
        const id_fornecedor = req.params.id_fornecedor;
        const response =
          await this.#pedidos.getUmPedidoComItensDeForncedorAprovado(
            id_pedido,
            id_fornecedor,
          );

        if (response.error) return res.status(response.code).json(response);
        return res.status(200).json(response);
      } catch (error) {
        console.error(error);
        return res.status(error.code || 500).json(error);
      }
    };
  }

  getPedidosAprovados() {
    return async (req, res) => {
      try {
        const response = await this.#pedidos.getPedidosAprovados();
        if (response.error) return res.status(response.code).json(response);
        return res.status(200).json(response);
      } catch (error) {
        console.error(error);
        return res.status(error.code || 500).json(error);
      }
    };
  }

  getItensAprovados() {
    return async (req, res) => {
      try {
        const id_pedido = req.params.id_pedido;
        const response = await this.#pedidos.getItensAprovados(id_pedido);

        if (response.error) return res.status(response.code).json(response);

        return res.status(200).json(response);
      } catch (error) {
        console.error(error);
        return res.status(error.code || 500).json(error);
      }
    };
  }

  responderCotacao() {
    return async (req, res) => {
      try {
        const data = req.body;

        const response = await this.#pedidos.responderCotacao(data);

        if (response.error) return res.status(response.code).json(response);
        return res.status(200).json(response);
      } catch (error) {
        console.error(error);
        return res.status(error.code || 500).json(error);
      }
    };
  }

  getPedidosDeFornecedor() {
    return async (req, res) => {
      try {
        const id_fornecedor = req.params.id_fornecedor;

        const response =
          await this.#pedidos.getPedidosDeFornecedor(id_fornecedor);

        if (response.error) return res.status(response.code).json(response);
        return res.status(200).json(response);
      } catch (error) {
        console.error(error);
        return res.status(error.code || 500).json(error);
      }
    };
  }

  getPedidosDeFornecedorAprovado() {
    return async (req, res) => {
      try {
        const id_fornecedor = req.params.id_fornecedor;

        const response =
          await this.#pedidos.getPedidosDeFornecedorAprovado(id_fornecedor);

        if (response.error) return res.status(response.code).json(response);
        return res.status(200).json(response);
      } catch (error) {
        console.error(error);
        return res.status(error.code || 500).json(error);
      }
    };
  }

  getPedidosGestor() {
    return async (req, res) => {
      try {
        const id_pedido = req.params.id_pedido;

        const response = await this.#pedidos.getPedidosGestor(id_pedido);

        if (response.error) return res.status(response.code).json(response);
        return res.status(200).json(response);
      } catch (error) {
        console.error(error);
        return res.status(error.code || 500).json(error);
      }
    };
  }

  /**
   * Atualiza um operário
   */
  update() {
    return async (req, res) => {
      try {
        const data = req.body;
        const response = await this.#pedidos.update(data);

        if (response.error) return res.status(response.code).json(response);
        return res.status(response.code).json(response);
      } catch (error) {
        console.error(error);
        return res.status(error.code || 500).json(error);
      }
    };
  }

  cotacao() {
    return async (req, res) => {
      try {
        const data = req.body;
        const response = await this.#pedidos.cotacao(data);

        if (response.error) return res.status(response.code).json(response);
        return res.status(response.code).json(response);
      } catch (error) {
        console.error(error);
        return res.status(error.code || 500).json(error);
      }
    };
  }

  enviar() {
    return async (req, res) => {
      try {
        const data = req.body;
        const response = await this.#pedidos.enviar(data);
        if (response.error) return res.status(response.code).json(response);
        return res.status(response.code).json(response);
      } catch (error) {
        console.error(error);
        return res.status(error.code || 500).json(error);
      }
    };
  }

  getTransito() {
    return async (req, res) => {
      try {
        const response = await this.#pedidos.getTransito();

        // Se você quiser checar error vindo do service
        if (response?.error) {
          return res.status(response.code || 400).json(response);
        }

        // SUCCESS → usa 200
        return res.status(200).json(response);
      } catch (error) {
        console.error(error);
        return res.status(error.code || 500).json(error);
      }
    };
  }

  getItensTransito() {
    return async (req, res) => {
      try {
        const id_pedido = req.params.id_pedido;
        const response = await this.#pedidos.getItensTransito(id_pedido);

        if (response.error) return res.status(response.code).json(response);
        return res.status(200).json(response);
      } catch (error) {
        console.error(error);
        return res.status(error.code || 500).json(error);
      }
    };
  }

  deletarCotacao() {
    return async (req, res) => {
      try {
        const id = req.params.id;
        const response = await this.#pedidos.deletarCotacao(id);
        if (response.error) return res.status(response.code).json(response);
        return res.status(response.code).json(response);
      } catch (error) {
        console.error(error);
        return res.status(error.code || 500).json(error);
      }
    };
  }

  alterarStatusPedido() {
    return async (req, res) => {
      try {
        const data = req.body;
        const response = await this.#pedidos.alterarStatusPedido(data);

        if (response.error) return res.status(response.code).json(response);
        return res.status(response.code).json(response);
      } catch (error) {
        console.error(error);
        return res.status(error.code || 500).json(error);
      }
    };
  }

  salvarDecisoesPedido() {
    return async (req, res) => {
      try {
        const data = req.body;
        const response = await this.#pedidos.salvarDecisoesPedido(data);

        // Status code padrão (200 para sucesso / 400 para erro)
        const status = response.code || (response.error ? 400 : 200);

        return res.status(status).json(response);
      } catch (error) {
        console.error(error);

        return res.status(error.code || 500).json({
          error: true,
          message: error.message || "Erro interno no servidor",
        });
      }
    };
  }

  recebimentoConfirmado() {
    return async (req, res) => {
      try {
        // O 'multer' coloca os campos de texto no body e os arquivos em files
        const dadosJson = req.body.dados;
        const arquivos = req.files;

        // Validação básica
        if (!dadosJson) {
          return res.status(400).json({
            error: true,
            message: "Dados do pedido não encontrados (campo 'dados').",
          });
        }

        // Passamos a string JSON e o array de arquivos para o serviço
        const response = await this.#pedidos.recebimentoConfirmado(
          dadosJson,
          arquivos,
        );

        if (response.error) {
          return res.status(response.code).json(response);
        }

        return res.status(response.code).json(response);
      } catch (error) {
        console.error("Erro no controller:", error);
        // Fallback para caso o erro não tenha código definido
        return res.status(error.code || 500).json({
          error: true,
          message: error.message || "Erro interno do servidor",
          code: error.code || 500,
        });
      }
    };
  }

  getPendenteEntrega() {
    return async (req, res) => {
      try {
        const response = await this.#pedidos.getPendenteEntrega();
        if (response.error) return res.status(response.code).json(response);
        return res.status(200).json(response);
      } catch (error) {
        console.error(error);
        return res.status(error.code || 500).json(error);
      }
    };
  }

  getItensPendentesEntrega() {
    return async (req, res) => {
      try {
        const id_pedido = req.params.id_pedido;
        const response =
          await this.#pedidos.getItensPendentesEntrega(id_pedido);
        return res.status(200).json(response);
      } catch (error) {
        console.error(error);
        return res.status(error.code || 500).json(error);
      }
    };
  }

  cancelaItemOuReenvia() {
    return async (req, res) => {
      try {
        const data = req.body;
        const id_item = req.params.id_item;
        const response = await this.#pedidos.cancelaItemOuReenvia(
          id_item,
          data,
        );
        return res.status(200).json(response);
      } catch (error) {
        console.error(error);
        return res.status(error.code || 500).json(error);
      }
    };
  }

  /**
   * Remove um operário
   */
  delete() {
    return async (req, res) => {
      try {
        const id = req.params.id;
        const response = await this.#pedidos.delete(id);

        if (response.error) return res.status(response.code).json(response);
        return res.status(response.code).json(response);
      } catch (error) {
        console.error(error);
        return res.status(error.code || 500).json(error);
      }
    };
  }
}

module.exports = PedidosController;

// a
