const { S3Client, PutObjectCommand } = require("@aws-sdk/client-s3");
const { response } = require("express");
const { sendMessage } = require("../../util/gzappy");

class OrçamentosRepository {
  #db = require("../../config/db");

  constructor() {
    this.s3Client = new S3Client({
      region: process.env.AWS_S3_REGION,
      credentials: {
        accessKeyId: process.env.AWS_ACCESS_KEY_ID,
        secretAccessKey: process.env.AWS_SECRET_ACCESS_KEY,
      },
    });

    this.S3_BUCKET = process.env.AWS_S3_BUCKET;
    this.region = process.env.AWS_S3_REGION;
  }

  register(data, file) {
    console.log(data);

    return new Promise(async (resolve, reject) => {
      try {
        let arquivo = null;

        // Upload no S3
        if (file) {
          const uniqueKey = `${Date.now()}_${file.originalname}`;
          arquivo = `https://${this.S3_BUCKET}.s3.${this.region}.amazonaws.com/${uniqueKey}`;

          const command = new PutObjectCommand({
            Bucket: this.S3_BUCKET,
            Key: uniqueKey,
            Body: file.buffer,
            ContentType: file.mimetype,
          });

          await this.s3Client.send(command);
        }

        // INSERT
        const insertSql = `
        INSERT INTO orcamentos
        (titulo, valor, validade_proposta, forma_pagamento, periodo_execucao, descricao, urls, criado_em, 
        atualizado_em, status, id_user, chave_pix, metodo, banco, favorecido_nome, doc_favorecido, conta, agencia
        )
        VALUES (?, ?, ?, ?, ?, ?, ?, NOW(), NOW(), 'pendente_gestor', ?, ?, ?, ?, ?, ?, ?, ?)
      `;

        this.#db.query(
          insertSql,
          [
            data.titulo,
            data.valor,
            data.validade,
            data.forma_pagamento,
            data.periodo_obra,
            data.descricao,
            arquivo,
            data.userId,
            data.financeiro.chave_pix,
            data.financeiro.metodo,
            data.financeiro.banco,
            data.financeiro.favorecido_nome,
            data.financeiro.doc_favorecido,
            data.financeiro.conta,
            data.financeiro.agencia,
          ],
          async (error, insertResult) => {
            if (error) {
              console.error(error);
              return reject({
                error: true,
                message: "Erro ao cadastrar orçamento",
                code: 500,
              });
            }

            // Buscar nome da empresa
            const selectSql = `SELECT nome FROM empresas WHERE id = ?`;

            this.#db.query(selectSql, [data.userId], (error, empresaResult) => {
              if (error) {
                console.error(error);
                return reject({
                  error: true,
                  message: "Erro ao buscar nome da empresa",
                  code: 500,
                });
              }

              const telefoneGestorSql = `SELECT telefone FROM admins`;

              this.#db.query(telefoneGestorSql, (error, telefoneResult) => {
                if (error) {
                  console.error(error);
                  return reject({
                    error: true,
                    message: "Erro ao buscar telefone do gestor",
                    code: 500,
                  });
                }

                //Envia mensagem para cada gestor encontrado
                telefoneResult.forEach((gestor) => {
                  sendMessage(
                    `A empresa ${empresaResult[0]?.nome} enviou um novo orçamento para sua avaliação.

Acesse o link abaixo para conferir os detalhes: https://obracroissantecia.com.br

Em caso de dúvidas, entre em contato com (11) 97259-7464. Mensagem automatica, nao responder.`,
                    gestor.telefone
                  );
                });

                return resolve({
                  success: true,
                  message: "Orçamento cadastrado com sucesso!",
                  empresa: empresaResult[0]?.nome || null,
                  code: 200,
                });
              });
            });
          }
        );
      } catch (err) {
        console.error("Erro ao cadastrar orçamento", err);
        return reject({
          error: true,
          message: "Erro ao cadastrar orçamento",
          code: 500,
        });
      }
    });
  }

  getAll() {
    return new Promise((resolve, reject) => {
      try {
        this.#db.query(
          `SELECT orcamentos.*, empresas.nome AS empresa FROM orcamentos, empresas 
                    WHERE
                    orcamentos.id_user = empresas.id`,
          (error, response) => {
            if (error) {
              console.error("Erro ao buscar orçamentos", error);
              return reject({
                error: true,
                message: "Erro ao buscar orçamentos",
                code: 500,
              });
            }
            return resolve(response);
          }
        );
      } catch (error) {
        console.error(error);
        reject({
          error: true,
          message: "Erro interno ao buscar orçamentos",
          code: 500,
        });
      }
    });
  }

  getAllByEmpresa(id_empresa) {
    return new Promise((resolve, reject) => {
      try {
        this.#db.query(
          `SELECT * FROM orcamentos WHERE id_user = ?`,
          [id_empresa],
          (error, response) => {
            if (error) {
              console.error(
                `Erro ao buscar orçamentos para a empresa ${id_empresa}`,
                error
              );
              return reject({
                error: true,
                message: "Erro ao buscar orçamentos por empresa",
                code: 500,
              });
            }
            return resolve(response);
          }
        );
      } catch (error) {
        console.error(error);
        reject({
          error: true,
          message: "Erro interno ao buscar orçamentos por empresa",
          code: 500,
        });
      }
    });
  }

  getById(id) {
    return new Promise((resolve, reject) => {
      try {
        this.#db.query(
          `SELECT * FROM orcamentos WHERE id = ?`,
          [id],
          (error, response) => {
            if (error) {
              console.error(
                `Erro ao buscar orçamentos para a empresa ${id}`,
                error
              );
              return reject({
                error: true,
                message: "Erro ao buscar orçamentos por empresa",
                code: 500,
              });
            }
            return resolve(response);
          }
        );
      } catch (error) {
        console.error(error);
        reject({
          error: true,
          message: "Erro interno ao buscar orçamentos por empresa",
          code: 500,
        });
      }
    });
  }

  updateStatus(data, id) {
    return new Promise((resolve, reject) => {
      try {
        // -------------------------------------------------------------
        // 📌 HELPER 1: Busca Título e ID da Empresa (Dono do orçamento)
        // -------------------------------------------------------------
        const getDadosOrcamento = (callback) => {
          // Buscamos o titulo E o id_user (que é o ID da empresa)
          const sql = `SELECT titulo, id_user FROM orcamentos WHERE id = ?`;

          this.#db.query(sql, [id], (error, result) => {
            if (error) {
              console.error(error);
              return reject({
                error: true,
                message: "Erro ao buscar dados do orçamento",
                code: 500,
              });
            }
            const dados = result[0] || {
              titulo: "(Sem título)",
              id_user: null,
            };
            callback(dados);
          });
        };

        // -------------------------------------------------------------
        // 📌 HELPER 2: Busca Contatos da Empresa e Envia MSG
        // -------------------------------------------------------------
        const notificarEmpresa = (empresaId, mensagem, callbackFinal) => {
          if (!empresaId) return callbackFinal();

          const sql = `
          SELECT responsavel1_celular AS r1, responsavel2_celular AS r2 
          FROM empresas WHERE id = ?
        `;

          this.#db.query(sql, [empresaId], (error, result) => {
            if (error) {
              console.error("Erro ao buscar contatos:", error);
              // Não rejeitamos aqui para não travar o fluxo principal, apenas logamos
              return callbackFinal();
            }

            const r1 = result[0]?.r1;
            const r2 = result[0]?.r2;

            if (r1) sendMessage(mensagem, r1);
            if (r2) sendMessage(mensagem, r2);

            callbackFinal();
          });
        };

        // =============================================================
        // 📌 1. STATUS = APROVADO
        // =============================================================
        if (data.status === "aprovado") {
          const updateOrcamentoSql = `UPDATE orcamentos SET status = ?, atualizado_em = NOW(), aprovado_em = NOW() WHERE id = ?`;

          this.#db.query(updateOrcamentoSql, [data.status, id], (error) => {
            if (error) {
              console.error(error);
              return reject({
                error: true,
                message: "Erro ao atualizar orçamento",
                code: 500,
              });
            }

            // No aprovado, também ativamos a empresa (lógica específica deste status)
            // Nota: Usamos getDadosOrcamento para pegar o ID da empresa correto do banco
            getDadosOrcamento((dadosOrcamento) => {
              const updateEmpresaSql = `UPDATE empresas SET status = 'ativo' WHERE id = ?`;

              this.#db.query(
                updateEmpresaSql,
                [dadosOrcamento.id_user],
                (errEmp) => {
                  if (errEmp) console.error("Erro ao ativar empresa", errEmp);

                  // Notifica
                  const msg = `O gestor aprovou o orçamento “${dadosOrcamento.titulo}”.

Acesse o link abaixo para visualizar os detalhes e acompanhar o andamento: https://obracroissantecia.com.br

Em caso de dúvidas, entre em contato com (11) 97259-7464. Mensagem automatica, nao responder.`;
                  notificarEmpresa(dadosOrcamento.id_user, msg, () => {
                    return resolve({
                      success: true,
                      message: "Orçamento aprovado!",
                      code: 200,
                    });
                  });
                }
              );
            });
          });
          return;
        }

        // =============================================================
        // 📌 2. STATUS = RECUSADO
        // =============================================================
        if (data.status === "recusado") {
          const sql = `UPDATE orcamentos SET status = ?, atualizado_em = NOW() WHERE id = ?`;

          this.#db.query(sql, [data.status, id], (error) => {
            if (error)
              return reject({
                error: true,
                message: "Erro ao atualizar orçamento",
                code: 500,
              });

            getDadosOrcamento((dadosOrcamento) => {
              const msg = `O gestor recusou o orçamento “${dadosOrcamento.titulo}”.

Acesse o link abaixo para visualizar o motivo da recusa: https://obracroissantecia.com.br

Em caso de dúvidas, entre em contato com (11) 97259-7464. Mensagem automatica, nao responder.`;

              // Busca contatos no banco e envia
              notificarEmpresa(dadosOrcamento.id_user, msg, () => {
                return resolve({
                  success: true,
                  message: "Status atualizado (Recusado)!",
                  code: 200,
                });
              });
            });
          });
          return;
        }

        // =============================================================
        // 📌 3. STATUS = PENDENTE EMPRESA
        // =============================================================
        if (data.status === "pendente_empresa") {
          const sql = `
          UPDATE orcamentos 
          SET valor = ?, justificativa = ?, status = ?, atualizado_em = NOW() 
          WHERE id = ?
        `;

          this.#db.query(
            sql,
            [data.orcamento, data.justificativa, data.status, id],
            (error) => {
              if (error)
                return reject({
                  error: true,
                  message: "Erro ao atualizar orçamento",
                  code: 500,
                });

              getDadosOrcamento((dadosOrcamento) => {
                // Nota: Ajuste a mensagem conforme a lógica de quem envia para quem
                const msg = `O orçamento “${dadosOrcamento.titulo}” foi atualizado e aguarda sua aprovação.

Acesse o link abaixo para acompanhar os detalhes: https://obracroissantecia.com.br

Em caso de dúvidas, entre em contato com (11) 97259-7464. Mensagem automatica, nao responder.`;

                // Busca contatos no banco e envia
                notificarEmpresa(dadosOrcamento.id_user, msg, () => {
                  return resolve({
                    success: true,
                    message: "Status atualizado (Pendente Empresa)!",
                    code: 200,
                  });
                });
              });
            }
          );
          return;
        }
      } catch (error) {
        console.error(error);
        reject({ error: true, message: "Erro interno", code: 500 });
      }
    });
  }

  statusUpdateEmpresa(data, id) {
    return new Promise((resolve, reject) => {
      try {
        // --------------------------------------------------
        // 📌 HELPER 1: Busca Título do Orçamento
        // --------------------------------------------------
        const getTitulo = (callback) => {
          const tituloSql = `SELECT titulo FROM orcamentos WHERE id = ?`;
          this.#db.query(tituloSql, [id], (error, result) => {
            if (error) {
              console.error("Erro ao buscar título:", error);
              return callback("(Sem título)"); // Retorna fallback em vez de falhar tudo
            }
            const titulo = result[0]?.titulo || "(Sem título)";
            callback(titulo);
          });
        };

        // --------------------------------------------------
        // 📌 HELPER 2: Busca Gestores e Envia Mensagem
        // --------------------------------------------------
        const notificarGestores = (mensagem, callbackFinal) => {
          const gestoresSql = `SELECT telefone FROM admins`;

          this.#db.query(gestoresSql, (error, result) => {
            if (error) {
              console.error("Erro ao buscar gestores para notificar:", error);
              // Não rejeitamos a Promise principal apenas por erro no envio da msg
              return callbackFinal();
            }

            if (result && result.length > 0) {
              result.forEach((gestor) => {
                sendMessage(mensagem, gestor.telefone);
              });
            }

            callbackFinal();
          });
        };

        // ==================================================
        // 📌 1. STATUS = APROVADO PELA EMPRESA
        // ==================================================
        if (data.status === "aprovado") {
          const updateSql = `UPDATE orcamentos SET status = ?, atualizado_em = NOW(), aprovado_em = NOW() WHERE id = ?`;

          this.#db.query(updateSql, [data.status, id], (error) => {
            if (error)
              return reject({
                error: true,
                message: "Erro ao atualizar status",
                code: 500,
              });

            getTitulo((titulo) => {
              const msg = `A empresa aprovou o orçamento “${titulo}”!

Acesse o link abaixo para visualizar os detalhes e acompanhar o andamento: https://obracroissantecia.com.br

Em caso de dúvidas, entre em contato com (11) 97259-7464. Mensagem automatica, nao responder.`;

              notificarGestores(msg, () => {
                return resolve({
                  success: true,
                  message: "Orçamento aprovado e gestores notificados.",
                  code: 200,
                });
              });
            });
          });
          return;
        }

        // ==================================================
        // 📌 2. STATUS = RECUSADO
        // ==================================================
        if (data.status === "recusado") {
          const sql = `UPDATE orcamentos SET status = ?, atualizado_em = NOW() WHERE id = ?`;

          this.#db.query(sql, [data.status, id], (error) => {
            if (error)
              return reject({
                error: true,
                message: "Erro ao atualizar status",
                code: 500,
              });

            getTitulo((titulo) => {
              const msg = `A empresa recusou o orçamento “${titulo}”.

Acesse o link abaixo para visualizar os detalhes e verificar o motivo da recusa:
https://obracroissantecia.com.br`;

              notificarGestores(msg, () => {
                return resolve({
                  success: true,
                  message: "Orçamento recusado e gestores notificados.",
                  code: 200,
                });
              });
            });
          });
          return;
        }

        // ==================================================
        // 📌 3. STATUS = PENDENTE GESTOR (CONTRA-PROPOSTA)
        // ==================================================
        if (data.status === "pendente_gestor") {
          const sql = `
          UPDATE orcamentos
          SET valor = ?, justificativa = ?, status = ?, atualizado_em = NOW()
          WHERE id = ?
        `;

          this.#db.query(
            sql,
            [data.orcamento, data.justificativa, data.status, id],
            (error) => {
              if (error)
                return reject({
                  error: true,
                  message: "Erro ao atualizar status",
                  code: 500,
                });

              getTitulo((titulo) => {
                const msg = `A empresa enviou uma nova contraproposta para o orçamento “${titulo}”.

Acesse o link abaixo para avaliar a proposta e acompanhar o andamento: https://obracroissantecia.com.br

Em caso de dúvidas, entre em contato com (11) 97259-7464. Mensagem automatica, nao responder.`;

                notificarGestores(msg, () => {
                  return resolve({
                    success: true,
                    message: "Contra-proposta enviada e gestores notificados.",
                    code: 200,
                  });
                });
              });
            }
          );
          return;
        }
      } catch (error) {
        console.error("Erro no try/catch geral:", error);
        return reject({ error: true, message: "Erro interno", code: 500 });
      }
    });
  }

  mudarPreçoCompras(data, id) {
    return new Promise((resolve, reject) => {
      try {
        this.#db.query(
          `UPDATE orcamentos SET valor = ? WHERE id = ?`,
          [data.valor, id],
          (error) => {
            if (error) {
              console.error("Erro ao atualizar preço", error);
              return reject({
                error: true,
                message: "Erro ao atualizar preço",
                code: 500,
              });
            }
            return resolve({
              success: true,
              message: "Preço atualizado com sucesso",
              code: 200,
            });
          }
        );
      } catch (error) {
        console.error(error);
        return reject({ error: true, message: "Erro interno", code: 500 });
      }
    });
  }
}

module.exports = OrçamentosRepository;
