const path = require("path");
const fs = require("fs");
const { validarCNPJ } = require("../../util/validarCNPJ");
const { validarCPF } = require("../../util/validarCPF");
const { PutObjectCommand, S3Client } = require("@aws-sdk/client-s3");

class OperariosRepository {
  #db = require("../../config/db");
  #region = process.env.AWS_S3_REGION;

  constructor() {
        this.s3Client = new S3Client({
      region: process.env.AWS_S3_REGION,
      credentials: {
        accessKeyId: process.env.AWS_ACCESS_KEY_ID,
        secretAccessKey: process.env.AWS_SECRET_ACCESS_KEY,
      },
    });
    this.S3_BUCKET = process.env.AWS_S3_BUCKET
  }

  /**
   * Cadastra um novo operário
   * @param {*} data
   */

register(data, files) {
  return new Promise(async (resolve, reject) => {
    try {

      let fotoDocumentoUrl = "";
      let fotoCursosUrl = "";
      let fotoEpiUrl = "";

      const uploadToS3 = async (file) => {
        if (!file) return "";

        const imageUrl = `https://${this.S3_BUCKET}.s3.${this.#region}.amazonaws.com/${file.originalname}`;

        const command = new PutObjectCommand({
          Bucket: this.S3_BUCKET,
          Key: file.originalname,
          Body: file.buffer,
          ContentType: file.mimetype
        });

        await this.s3Client.send(command);

        return imageUrl;
      };
      // id_empresa

      fotoDocumentoUrl = await uploadToS3(files?.foto_documento?.[0]);
      fotoCursosUrl    = await uploadToS3(files?.foto_cursos?.[0]);
      fotoEpiUrl       = await uploadToS3(files?.foto_epi?.[0]);

      this.#db.query(
        `INSERT INTO operarios 
         (nome, celular, foto_documento, foto_cursos, foto_epi, horario_entrada, horario_saida, data_registro)
         VALUES (?, ?, ?, ?, ?, ?, ?, now())`,
        [
          data.nome,
          data.celular || "",
          fotoDocumentoUrl,
          fotoCursosUrl,
          fotoEpiUrl,
          data.horario_entrada,
          data.horario_saida,
        ],
        (error, response) => {
          
          const idOperario = response.insertId;

          if (data.id_empresa) {
            this.#db.query(
              `INSERT INTO operarios_empresa (id_empresa, id_operario)
               VALUES (?, ?)`,
              [data.id_empresa, idOperario],
              (errorRelacionamento) => {
                if (errorRelacionamento) {
                  console.error("Erro ao vincular operário à empresa", errorRelacionamento);
                  return reject({
                    error: true,
                    message: "Operário cadastrado, mas erro ao vincular empresa",
                    code: 500,
                  });
                }

                return resolve({
                  success: true,
                  message: "Operário cadastrado e vinculado à empresa com sucesso!",
                  code: 200,
                });
              }
            );
          } else {
            // Caso não venha id_empresa
            return resolve({
              success: true,
              message: "Operário cadastrado com sucesso (sem empresa vinculada)",
              code: 200,
            });
          }
        }
      );

    } catch (error) {
      console.error("Erro ao cadastrar operário ", error);
      return reject({
        error: true,
        message: "Erro ao cadastrar operário",
        code: 500
      });
    }
  });
}


  /**
   * Lista todos os operários
   */
  getOperarios() {
    return new Promise((resolve, reject) => {
      try {
        this.#db.query(
          `SELECT id, nome, celular, foto_documento, foto_cursos, foto_epi, horario_entrada, horario_saida
           FROM operarios`,
          [],
          (error, operarios) => {
            if (error) {
              console.error("Erro ao buscar operários ", error);
              return reject({
                error: true,
                message: "Erro ao buscar operários",
                code: 500,
              });
            }
            return resolve(operarios);
          }
        );
      } catch (error) {
        console.error("Erro ao buscar operários ", error);
        return reject({
          error: true,
          message: "Erro ao buscar operários",
          code: 500,
        });
      }
    });
  }

    getOperariosPorData(data) {
    return new Promise((resolve, reject) => {
      try {
        this.#db.query(
          `SELECT 
              o.id,
              o.nome,
              o.celular,
              o.foto_documento,
              o.foto_cursos,
              o.foto_epi,
              o.horario_entrada,
              o.horario_saida,
              DATE_FORMAT(o.data_registro, "%d/%m/%Y") AS data_registros,
              IFNULL(e.nome, '') AS nome_empresa
          FROM operarios o
          LEFT JOIN operarios_empresa oe ON oe.id_operario = o.id
          LEFT JOIN empresas e ON e.id = oe.id_empresa
          WHERE o.data_registro = ?`,
          [data],
          (error, operarios) => {
            if (error) {
              console.error("Erro ao buscar operários ", error);
              return reject({ error: true, message: "Erro ao buscar operários", code: 500 });
            }
            return resolve(operarios);
          }
        );
      } catch (error) {
        console.error("Erro ao buscar operários ", error);
        return reject({ error: true, message: "Erro ao buscar operários", code: 500});
      }
    });
  }

  getOperariosDeHojePorEmpresa(id_empresa) {
    return new Promise((resolve, reject) => {
      try {
        this.#db.query(
          `SELECT operarios.id, operarios.nome, operarios.celular, operarios.foto_documento, operarios.foto_cursos, operarios.foto_epi, operarios.horario_entrada, operarios.horario_saida, 
            DATE_FORMAT(operarios.data_registro, "%d/%m/%Y") AS data_registros, empresas.id AS id_empresa, empresas.nome AS nome_empresa
            FROM operarios, operarios_empresa, empresas
            WHERE 
            operarios_empresa.id_operario = operarios.id AND
            operarios_empresa.id_empresa = empresas.id AND
            operarios.data_registro = CURDATE() AND
            operarios_empresa.id_empresa = ?;`,
          [id_empresa],
          (error, operarios) => {
            if (error) {
              console.error("Erro ao buscar operários ", error);
              return reject({ error: true, message: "Erro ao buscar operários", code: 500 });
            }

            if (operarios.length === 0) {
              return reject({
                error: true,
                message: `Nenhum operario cadastrado hoje`,
                code: 404
              })
            }
            return resolve(operarios);
          }
        );
      } catch (error) {
        console.error("Erro ao buscar operários ", error);
        return reject({ error: true, message: "Erro ao buscar operários", code: 500 });
      }
    });
  }

  /**
   * Busca um único operário
   * @param {*} id 
   */
  getUmOperario(id) {
    return new Promise((resolve, reject) => {
      try {
        this.#db.query(
          `SELECT id, nome, celular, foto_documento, foto_cursos, foto_epi, horario_entrada, horario_saida
           FROM operarios
           WHERE id = ?`,
          [id],
          (error, results) => {
            if (error) {
              console.error("Erro ao buscar operário ", error);
              return reject({
                error: true,
                message: "Erro ao buscar operário",
                code: 500,
              });
            }

            if (results.length === 0) {
              return reject({
                error: true,
                message: "Operário não encontrado",
                code: 404,
              });
            }

            return resolve(results[0]);
          }
        );
      } catch (error) {
        console.error("Erro ao buscar operário ", error);
        return reject({
          error: true,
          message: "Erro ao buscar operário",
          code: 500,
        });
      }
    });
  }

  /**
   * Deleta operário
   */
  delete(id) {
    return new Promise((resolve, reject) => {
      this.#db.query(
        `DELETE FROM operarios WHERE id = ?`,
        [id],
        (error) => {
          if (error) {
            console.error("Erro ao deletar operário ", error);
            return reject({
              error: true,
              message: "Erro ao deletar operário",
              code: 500,
            });
          }

          return resolve({
            success: true,
            message: "Operário deletado com sucesso!",
            code: 200,
          });
        }
      );
    });
  }

  /**
   * Atualiza um operário
   * @param {Object} data 
   */
  update(data) {
    return new Promise(async (resolve, reject) => {
      try {
        const fields = [];
        const values = [];

        if (data.nome) { fields.push("nome = ?"); values.push(data.nome); }
        if (data.celular) { fields.push("celular = ?"); values.push(data.celular); }
        if (data.foto_documento) { fields.push("foto_documento = ?"); values.push(data.foto_documento); }
        if (data.foto_cursos) { fields.push("foto_cursos = ?"); values.push(data.foto_cursos); }
        if (data.foto_epi) { fields.push("foto_epi = ?"); values.push(data.foto_epi); }
        if (data.horario_entrada) { fields.push("horario_entrada = ?"); values.push(data.horario_entrada); }
        if (data.horario_saida) { fields.push("horario_saida = ?"); values.push(data.horario_saida); }

        if (fields.length === 0) {
          return reject({ error: true, message: "Nenhum dado para atualizar", code: 422 });
        }

        values.push(data.id);

        const query = `
          UPDATE operarios
          SET ${fields.join(", ")}
          WHERE id = ?
        `;

        this.#db.query(query, values, (error, response) => {
          if (error) {
            console.error("Erro ao atualizar operário ", error);
            return reject({
              error: true,
              message: "Erro ao atualizar operário",
              code: 500,
            });
          }

          if (response.affectedRows === 0) {
            return reject({
              error: true,
              message: "Operário não encontrado",
              code: 404,
            });
          }

          return resolve({
            success: true,
            message: "Operário atualizado com sucesso!",
            code: 200,
          });
        });

      } catch (error) {
        console.error(error);
        return reject({
          error: true,
          message: "Erro ao atualizar operário",
          code: 500,
        });
      }
    });
  }
}

module.exports = OperariosRepository;
