const { genereteToken } = require("../../config/auth/jwt");
const nodemailer = require("nodemailer");
const path = require("path");
const https = require("https");
const fs = require("fs");
const bcrypt = require("bcrypt");
const { validarCNPJ } = require("../../util/validarCNPJ");
const { validarCPF } = require("../../util/validarCPF");
const { S3Client, PutObjectCommand } = require("@aws-sdk/client-s3");
const { sendMedia, sendImage, sendMessage } = require("../../util/gzappy");

class FornecedoresRepository {
  #db = require("../../config/db");
  // #region = process.env.AWS_S3_REGION;

  constructor() {
    this.s3Client = new S3Client({
      region: process.env.AWS_S3_REGION,
      credentials: {
        accessKeyId: process.env.AWS_ACCESS_KEY_ID,
        secretAccessKey: process.env.AWS_SECRET_ACCESS_KEY,
      },
    });
    this.S3_BUCKET = process.env.AWS_S3_BUCKET;
    this.region = process.env.AWS_S3_REGION;
  }

  /**
   *
   * @param {*} data
   * @returns
   */
  register(data) {
    return new Promise(async (resolve, reject) => {
      try {
        const resultado =
          data.cnpj_cpf.length === 11
            ? validarCPF(data.cnpj_cpf)
            : data.cnpj_cpf.length === 14
              ? validarCNPJ(data.cnpj_cpf)
              : null;

        if (!resultado) {
          return reject({
            error: true,
            message: "CNPJ ou CPF inválido!",
            code: 422,
          });
        }

        this.#db.query(
          "SELECT id FROM fornecedores WHERE cnpj_cpf = ?",
          [data.cnpj_cpf],
          async (error, results) => {
            if (error) {
              console.error(error);
              return reject({
                error: true,
                message: "Erro ao verificar fornecedor",
                code: 500,
              });
            }

            if (results.length > 0) {
              return reject({
                error: true,
                message: "CNPJ ou CPF já cadastrado!",
                code: 409,
              });
            }

            const hashedPassword = await bcrypt.hash(data.senha, 10);

            this.#db.query(
              `INSERT INTO fornecedores 
             (nome_empresa, nome, telefone, email, ramo_atividade, auth, senha, cnpj_cpf)
             VALUES (?, ?, ?, ?, ?, ?, ?, ?)`,
              [
                data.nome_empresa,
                data.nome,
                data.telefone || "",
                data.email || "",
                data.ramo_atividade || "",
                data.auth || "",
                hashedPassword,
                data.cnpj_cpf,
              ],
              async (error, response) => {
                if (error) {
                  console.error(error);
                  return reject({
                    error: true,
                    message: "Erro ao cadastrar fornecedor",
                    code: 500,
                  });
                }
                this.#db.query(
                  `INSERT INTO users (nome, senha, role, cnpj_cpf, trueId)
                 SELECT nome, senha, 'fornecedor', cnpj_cpf, id
                 FROM fornecedores
                 WHERE cnpj_cpf NOT IN (SELECT cnpj_cpf FROM users);`,
                  (error) => {
                    if (error) {
                      console.error(
                        "Erro ao cadastrar usuário fornecedor ",
                        error,
                      );
                      return reject({
                        error: true,
                        message: "Erro ao cadastrar usuário fornecedor",
                        code: 500,
                      });
                    }
                    this.#db.query(
                      `UPDATE fornecedores f
                    INNER JOIN users u ON f.cnpj_cpf = u.cnpj_cpf
                    SET f.id_user = u.id
                    WHERE u.role = 'fornecedor'`,
                      (error) => {
                        if (error) {
                          console.error("Erro ao atualizar fornecedor ", error);
                          return reject({
                            error: true,
                            message: "Erro ao atualizar fornecedor",
                            code: 500,
                          });
                        }
                      },
                    );
                  },
                );
                return resolve({
                  success: true,
                  message: "Fornecedor cadastrado com sucesso!",
                  code: 200,
                });
              },
            );
          },
        );
      } catch (err) {
        console.error("Erro ao cadastrar fornecedor ", err);
        return reject({
          error: true,
          message: "Erro ao cadastrar fornecedor",
          code: 500,
        });
      }
    });
  }

  enviarFormulario(data, file) {
    return new Promise(async (resolve, reject) => {
      try {
        const resultado =
          data.cnpj_cpf.length === 11
            ? validarCPF(data.cnpj_cpf)
            : data.cnpj_cpf.length === 14
              ? validarCNPJ(data.cnpj_cpf)
              : null;

        if (!resultado) {
          return reject({
            error: true,
            message: "CNPJ ou CPF inválido!",
            code: 422,
          });
        }

        this.#db.query(
          "SELECT id FROM fornecedores WHERE cnpj_cpf = ?",
          [data.cnpj_cpf],
          async (error, results) => {
            if (error) {
              console.error(error);
              return reject({
                error: true,
                message: "Erro ao verificar fornecedor",
                code: 500,
              });
            }

            if (results.length > 0) {
              return reject({
                error: true,
                message: "CNPJ ou CPF já cadastrado!",
                code: 409,
              });
            }

            let imageUrl;
            const hashedPassword = await bcrypt.hash(data.senha, 10);

            if (file) {
              imageUrl = `https://${this.S3_BUCKET}.s3.${this.region}.amazonaws.com/${file.originalname}`;

              const command = new PutObjectCommand({
                Bucket: this.S3_BUCKET,
                Key: file.originalname,
                Body: file.buffer,
                ContentType: file.mimetype,
              });

              await this.s3Client.send(command);
            }

            this.#db.query(
              `INSERT INTO fornecedores 
            (nome_empresa, nome, telefone, email, ramo_atividade, cnpj_cpf, senha, url_arquivo, status)
            VALUES (?, ?, ?, ?, ?, ?, ?, ?, 'ativo');
            `,
              [
                data.nome_empresa,
                data.nome,
                data.telefone || "",
                data.email || "",
                data.ramo_atividade || "",
                data.cnpj_cpf,
                hashedPassword,
                imageUrl,
              ],
              async (error, response) => {
                if (error) {
                  console.error("Erro ao cadastrar Fornecedor ", error);
                  return reject({
                    error: true,
                    message: "Erro ao cadastrar Fornecedor",
                    code: 500,
                  });
                }
                this.#db.query(
                  `INSERT INTO users (nome, senha, role, cnpj_cpf, trueId)
                 SELECT nome, senha, 'fornecedor', cnpj_cpf, id
                 FROM fornecedores
                 ;`,
                  (error) => {
                    if (error) {
                      console.error(
                        "Erro ao cadastrar usuário fornecedor ",
                        error,
                      );
                      return reject({
                        error: true,
                        message: "Erro ao cadastrar usuário fornecedor",
                        code: 500,
                      });
                    }
                    this.#db.query(
                      `UPDATE fornecedores f
                    INNER JOIN users u ON f.cnpj_cpf = u.cnpj_cpf
                    SET f.id_user = u.id
                    WHERE u.role = 'fornecedor'`,
                      (error) => {
                        if (error) {
                          console.error("Erro ao atualizar fornecedor ", error);
                          return reject({
                            error: true,
                            message: "Erro ao atualizar fornecedor",
                            code: 500,
                          });
                        }
                      },
                    );
                  },
                );

                //await sendMedia(`${data.reponsavel_contrato}`, "+55"+data.whats_envio, imageUrl);
                return resolve({
                  success: true,
                  message: "Fornecedor cadastrado com sucesso!",
                  code: 200,
                });
              },
            );
          },
        );
      } catch (error) {
        console.error("Erro ao cadastrar Fornecedor ", error);
        return reject({
          error: true,
          message: "Erro ao cadastrar Fornecedor",
          code: 500,
        });
      }
    });
  }

  /**
   * Lista todas as lojas
   */
  getFornecedores() {
    return new Promise((resolve, reject) => {
      try {
        this.#db.query(
          `SELECT id, nome_empresa, nome, telefone, email, ramo_atividade, role, cnpj_cpf, url_arquivo
         FROM fornecedores`,
          [],
          (error, fornecedores) => {
            if (error) {
              console.error("Erro ao buscar fornecedores ", error);
              return reject({
                error: true,
                message: "Erro ao buscar fornecedores",
                code: 500,
              });
            }
            return resolve(fornecedores);
          },
        );
      } catch (error) {
        console.error("Erro ao buscar fornecedores ", error);
        return reject({
          error: true,
          message: "Erro ao buscar fornecedores",
          code: 500,
        });
      }
    });
  }

  /**
   * Busca apenas um fornecedor
   * @param {*} id
   * @returns
   */
  getUmFornecedor(id) {
    return new Promise((resolve, reject) => {
      try {
        this.#db.query(
          `SELECT id, nome_empresa, nome, telefone, email, ramo_atividade, role, cnpj_cpf, url_arquivo
         FROM fornecedores
         WHERE id = ?`,
          [id],
          (error, results) => {
            if (error) {
              console.error("Erro ao buscar fornecedor ", error);
              return reject({
                error: true,
                message: "Erro ao buscar fornecedor",
                code: 500,
              });
            }

            if (results.length === 0) {
              return reject({
                error: true,
                message: "Fornecedor não encontrado",
                code: 404,
              });
            }

            return resolve(results[0]);
          },
        );
      } catch (error) {
        console.error("Erro ao buscar fornecedor ", error);
        return reject({
          error: true,
          message: "Erro ao buscar fornecedor",
          code: 500,
        });
      }
    });
  }

  /**
   * Deleta uma loja
   * @param {*} id
   * @returns
   */
  delete(id) {
    return new Promise((resolve, reject) => {
      this.#db.query(
        `UPDATE fornecedores SET status = 'inativo' WHERE id = ?`,
        [id],
        (error) => {
          if (error) {
            console.error("Erro ao deletar fornecedor ", error);
            return reject({
              error: true,
              message: "Erro ao deletar fornecedor",
              code: 500,
            });
          }
          return resolve({
            success: true,
            message: "Fornecedor deletado com sucesso!",
            code: 200,
          });
        },
      );
    });
  }

  /**
   * Atualiza um fornecedor
   * @param {Object} data
   * @returns Mensagem referente ao status da operação
   */
  update(data, file) {
    return new Promise(async (resolve, reject) => {
      try {
        const fieldsFornecedor = [];
        const valuesFornecedor = [];

        const fieldsUser = [];
        const valuesUser = [];

        let imageUrl;
        let hashedPassword;

        // ---- HASH DE SENHA ----
        if (data.senha) {
          hashedPassword = await bcrypt.hash(data.senha, 10);
        }

        // ---- UPLOAD PARA S3 ----
        if (file) {
          imageUrl = `https://${this.S3_BUCKET}.s3.${this.region}.amazonaws.com/${file.originalname}`;

          const command = new PutObjectCommand({
            Bucket: this.S3_BUCKET,
            Key: file.originalname,
            Body: file.buffer,
            ContentType: file.mimetype,
          });

          await this.s3Client.send(command);
        }

        // ---- CAMPOS DE FORNECEDORES ----
        if (data.nome_empresa) {
          fieldsFornecedor.push("nome_empresa = ?");
          valuesFornecedor.push(data.nome_empresa);
        }
        if (data.nome) {
          fieldsFornecedor.push("nome = ?");
          valuesFornecedor.push(data.nome);
        }
        if (data.telefone) {
          fieldsFornecedor.push("telefone = ?");
          valuesFornecedor.push(data.telefone);
        }
        if (data.email) {
          fieldsFornecedor.push("email = ?");
          valuesFornecedor.push(data.email);
        }
        if (data.ramo_atividade) {
          fieldsFornecedor.push("ramo_atividade = ?");
          valuesFornecedor.push(data.ramo_atividade);
        }

        if (hashedPassword) {
          fieldsFornecedor.push("senha = ?");
          valuesFornecedor.push(hashedPassword);
        }
        if (file && imageUrl) {
          fieldsFornecedor.push("url_arquivo = ?");
          valuesFornecedor.push(imageUrl);
        }

        if (fieldsFornecedor.length === 0) {
          return reject({
            error: true,
            message: "Nenhum dado para atualizar",
            code: 422,
          });
        }

        // ---- UPDATE DE FORNECEDORES ----
        valuesFornecedor.push(data.id);

        const queryFornecedor = `
        UPDATE fornecedores
        SET ${fieldsFornecedor.join(", ")}
        WHERE id = ?
      `;

        this.#db.query(queryFornecedor, valuesFornecedor, (error, response) => {
          if (error) {
            console.error(error);
            return reject({
              error: true,
              message: "Erro ao atualizar fornecedor",
              code: 500,
            });
          }

          if (response.affectedRows === 0) {
            return reject({
              error: true,
              message: "Fornecedor não encontrado",
              code: 404,
            });
          }

          // ---------------------------------------------
          //      CAMPOS DE USER (somente nome e senha)
          // ---------------------------------------------
          if (data.nome_empresa) {
            fieldsUser.push("nome = ?");
            valuesUser.push(data.nome_empresa);
          }

          if (hashedPassword) {
            fieldsUser.push("senha = ?");
            valuesUser.push(hashedPassword);
          }

          if (fieldsUser.length === 0) {
            return resolve({
              success: true,
              message: "Fornecedor atualizado com sucesso!",
              code: 200,
            });
          }

          valuesUser.push(data.id);

          const queryUser = `
          UPDATE users
          SET ${fieldsUser.join(", ")}
          WHERE trueId = ? AND role = 'fornecedor'
        `;

          this.#db.query(queryUser, valuesUser, (errorUser, respUser) => {
            if (errorUser) {
              console.error(errorUser);
              return reject({
                error: true,
                message: "Erro ao atualizar usuário vinculado ao fornecedor",
                code: 500,
              });
            }

            if (respUser.affectedRows === 0) {
              return reject({
                error: true,
                message: "Usuário do fornecedor não encontrado",
                code: 404,
              });
            }

            return resolve({
              success: true,
              message: "Fornecedor e usuário atualizados com sucesso!",
              code: 200,
            });
          });
        });
      } catch (error) {
        console.error(error);
        return reject({
          error: true,
          message: "Erro ao atualizar fornecedor",
          code: 500,
        });
      }
    });
  }
}

module.exports = FornecedoresRepository;
