const { genereteToken } = require("../../config/auth/jwt");
const nodemailer = require("nodemailer");
const path = require("path");
const https = require("https");
const fs = require("fs");
const bcrypt = require('bcrypt');
const { validarCNPJ } = require("../../util/validarCNPJ");
const { validarCPF } = require("../../util/validarCPF");
const { sendMessage } = require("../../util/gzappy");

class AuthRepository {
    #db = require("../../config/db");

    constructor() { }

    /**
     * 
     * @param {*} data 
     * @returns 
     */
    registerFinanceiro(data) {
        return new Promise(async (resolve, reject) => {
            try {

                const resultado =
                    data.cnpj_cpf.length === 11 ? validarCPF(data.cnpj_cpf) :
                        data.cnpj_cpf.length === 14 ? validarCNPJ(data.cnpj_cpf) : null;

                if (!resultado) {
                    return reject({ error: true, message: "CNPJ ou CPF inválido!", code: 422 });
                }

                const hashedPassword = await bcrypt.hash(data.senha, 10);

                this.#db.query(
                    `INSERT INTO financeiros (nome, email, cnpj_cpf, senha, celular) VALUES (?, ?, ?, ?, ?)`,
                    [data.nome, data.email, data.cnpj_cpf, hashedPassword, data.celular],
                    async (error, response) => {
                        if (error) {
                            console.error(error);
                            return reject({ error: true, message: "Erro ao cadastrar financeiro", code: 500 });
                        }

                        this.#db.query(
                            `INSERT INTO users (nome, cnpj_cpf, senha, role, trueId)
                         SELECT nome, cnpj_cpf, senha, 'financeiro', id
                         FROM financeiros
                         WHERE cnpj_cpf NOT IN (SELECT cnpj_cpf FROM users);`,
                            (error) => {
                                if (error) {
                                    console.error(error);
                                    return reject({ error: true, message: "Erro ao cadastrar usuário financeiro", code: 500 });
                                }

                                this.#db.query(
                                    `UPDATE financeiros f
                                 INNER JOIN users u ON f.cnpj_cpf = u.cnpj_cpf
                                 SET f.id_user = u.id
                                 WHERE u.role = 'financeiro';`,
                                    (error) => {
                                        if (error) {
                                            console.error(error);
                                            return reject({ error: true, message: "Erro ao atualizar financeiro", code: 500 });
                                        }
                                    }
                                );
                            }
                        );

                        return resolve({ success: true, message: "Financeiro cadastrado com sucesso!", code: 200 });
                    }
                );

            } catch (err) {
                console.error("Erro ao cadastrar financeiro ", err);
                return reject({ error: true, message: "Erro ao cadastrar financeiro", code: 500 });
            }
        });
    }



    updateEmpresa(data, id_empresa) {
        return new Promise(async (resolve, reject) => {
            try {
                const campos = [];
                const valores = [];

                // 1. Tratamento Especial da Senha
                // Se o usuário enviou uma nova senha, precisamos criptografá-la antes
                if (data.senha_inicial || data.senha) {
                    const senhaPlana = data.senha_inicial || data.senha;
                    const hash = await bcrypt.hash(senhaPlana, 10);

                    campos.push("senha = ?");
                    valores.push(hash);
                }

                // 2. Mapa de Tradução (Front-end -> Banco de Dados)
                // Baseado no seu INSERT: Chave do objeto 'data' : Coluna no Banco
                const mapaDeCampos = {
                    'razao_social': 'nome',
                    'telefone': 'telefone_empresa',
                    'resp1_nome': 'responsavel1_nome',
                    'resp1_celular': 'responsavel1_celular',
                    'resp1_email': 'responsavel1_email',
                    'resp2_nome': 'responsavel2_nome',
                    'resp2_celular': 'responsavel2_celular',
                    'resp2_email': 'responsavel2_email',
                    'exige_epi': 'exige_foto_epi',
                    // 'auth': null, // Geralmente não se atualiza auth manualmente assim
                    'contrato_periodo': 'contrato_periodo',
                    'contrato_valor': 'contrato_valor',
                    'contrato_pagamento': 'contrato_pagamento',
                    'contrato_tipo': 'contrato_tipo',
                    'contrato_descricao': 'contrato_descricao',
                    'whatsapp_notificacao_1': 'whatsapp_notificacao_1',
                    'whatsapp_notificacao_2': 'whatsapp_notificacao_2',
                    'atividade': 'atividade'
                };

                // 3. Loop Inteligente
                // Percorre o mapa. Se o dado existir no objeto 'data', adiciona na query.
                Object.keys(mapaDeCampos).forEach(campoData => {
                    const colunaBanco = mapaDeCampos[campoData];

                    // Verifica se o dado foi enviado e não é undefined
                    if (data[campoData] !== undefined) {
                        campos.push(`${colunaBanco} = ?`);
                        valores.push(data[campoData]);
                    }
                });

                // 4. Verificação de segurança
                if (campos.length === 0) {
                    return reject({
                        error: true,
                        message: "Nenhum dado válido enviado para atualização.",
                        code: 400
                    });
                }

                // 5. Montagem e Execução
                const sql = `UPDATE empresas SET ${campos.join(', ')} WHERE id = ?`;
                valores.push(id_empresa);

                this.#db.query(sql, valores, (error, response) => {
                    if (error) {
                        console.error("Erro SQL Update:", error);
                        return reject({
                            error: true,
                            message: "Erro interno ao atualizar empresa",
                            code: 500
                        });
                    }

                    if (response.affectedRows === 0) {
                        return reject({
                            error: true,
                            message: "Empresa não encontrada.",
                            code: 404
                        });
                    }

                    return resolve({
                        success: true,
                        message: "Dados atualizados com sucesso!",
                        code: 200
                    });
                });

            } catch (err) {
                console.error("Erro no updateEmpresa:", err);
                return reject({
                    error: true,
                    message: "Erro crítico ao atualizar.",
                    code: 500
                });
            }
        });
    }
    deleteEmpresa(id_empresa) {
        return new Promise((resolve, reject) => {
            try {
                this.#db.query(
                    `UPDATE empresas SET status = 'inativo' WHERE id = ?`,
                    [id_empresa],
                    (error, response) => {
                        if (error) {
                            console.error("Erro ao deletar empresa")
                            return reject({ error: true, message: "Erro ao desativar empresa", code: 500 })
                        }
                        return resolve({ success: true, message: "Empresa desativada com sucesso!", code: 200 });
                    }
                );
            } catch (error) {
                console.error("Erro ao deletar empresa ", error);
                return reject({ error: true, message: "Erro ao desativar empresa", code: 500 });
            }
        })
    }

    getEmpresas() {
        return new Promise((resolve, reject) => {
            try {
                this.#db.query(
                    `SELECT 
                        e.*,
                        (SELECT JSON_ARRAYAGG(JSON_OBJECT('id', o.id, 'valor', o.valor, 'descricao', o.descricao, 'validade_proposta', o.validade_proposta, 'forma_pagamento', o.forma_pagamento, 'periodo_execucao', o.periodo_execucao, 'urls', o.urls, 'criado_em', o.criado_em, 'atualizado_em', o.atualizado_em, 'status', o.status, 'justificativa', o.justificativa, 'titulo', o.titulo)) 
                         FROM orcamentos o 
                         WHERE o.id_user = e.id) as orcamentos
                    FROM empresas e`,
                    [],
                    (error, empresas) => {
                        if (error) {
                            console.error("Erro ao buscar empresas ", error);
                            return reject({ error: true, message: "Erro ao buscar empresas", code: 500 });
                        }

                        // O resultado de orcamentos é uma string JSON, vamos fazer o parse.
                        const empresasComOrcamentos = empresas.map(empresa => {
                            if (empresa.orcamentos) {
                                empresa.orcamentos = JSON.parse(empresa.orcamentos);
                            }
                            return empresa;
                        });
                        return resolve(empresasComOrcamentos);
                    }
                );
            } catch (error) {
                console.error("Erro ao buscar empresas ", error);
                return reject({ error: true, message: "Erro ao buscar empresas", code: 500 });
            }
        });
    }

    getStatusEmpresa(id_empresa) {
        return new Promise((resolve, reject) => {
            try {
                this.#db.query(
                    `SELECT status FROM empresas WHERE id = ?`,
                    [id_empresa],
                    (error, result) => {
                        if (error) {
                            console.error("Erro ao buscar status ", error);
                            return reject({ error: true, message: "Erro ao buscar status", code: 500 });
                        }
                        if (result.length === 0) {
                            return reject({ error: true, message: "Empresa não encontrada.", code: 404 });
                        }
                        return resolve(result[0]);
                    }
                );


            } catch (error) {
                console.error("Erro ao buscar status da empresa ", error);
                return reject({
                    error: true,
                    message: "Erro ao buscar status da empresa",
                    code: 500
                })
            }
        })
    }


    /**
     * Lista todas as lojas
     */
    getFinanceiros() {
        return new Promise((resolve, reject) => {
            try {
                this.#db.query(
                    `SELECT id, nome, cnpj_cpf, celular, email FROM financeiros`,
                    [],
                    (error, financeiros) => {
                        if (error) {
                            console.error("Erro ao buscar financeiros ", error);
                            return reject({ error: true, message: "Erro ao buscar financeiros", code: 500 });
                        }
                        return resolve(financeiros);
                    }
                );
            } catch (error) {
                console.error("Erro ao buscar financeiros ", error);
                return reject({ error: true, message: "Erro ao buscar financeiros", code: 500 });
            }
        });
    }

    /**
     * Busca apenas uma loja
     * @param {*} id 
     * @returns 
     */
    getUmFinanceiro(id) {
        return new Promise((resolve, reject) => {
            try {
                this.#db.query(
                    `SELECT id, nome, cnpj_cpf, celular, email FROM financeiros WHERE id = ?`,
                    [id],
                    (error, financeiros) => {
                        if (error) {
                            console.error("Erro ao buscar financeiros ", error);
                            return reject({ error: true, message: "Erro ao buscar financeiros", code: 500 });
                        }
                        if (financeiros.length === 0) {
                            return resolve({
                                error: true,
                                message: "Nenhum financeiros encontrada com esse ID",
                                code: 404
                            });
                        }
                        return resolve(financeiros[0]);
                    }
                );
            } catch (error) {
                console.error("Erro ao buscar financeiros ", error);
                return reject({ error: true, message: "Erro ao buscar financeiros", code: 500 });
            }
        });
    }

    getCelularesDosFinanceiros() {
        return new Promise((resolve, reject) => {
            try {
                this.#db.query(
                    `SELECT nome, celular FROM financeiros`, [],
                    (error, financeiros) => {
                        if (error) {
                            console.error("Erro ao buscar celular financeiros ", error);
                            return reject({ error: true, message: "Erro ao buscar celular financeiros", code: 500 });
                        }
                        // if (financeiros.length === 0) {
                        //     return resolve({
                        //         error: true,
                        //         message: "Nenhum celular dos financeiros encontrado",
                        //         code: 404
                        //     });
                        // }
                        return resolve(financeiros);
                    }
                );
            } catch (error) {
                console.error("Erro ao buscar financeiros ", error);
                return reject({ error: true, message: "Erro ao buscar financeiros", code: 500 });
            }
        });
    }

    enviarMensagemFinanceiro(data) {
    return new Promise(async (resolve, reject) => {
        try {

            if (!Array.isArray(data.destinatarios)) {
                return reject({
                    error: true,
                    message: "destinatarios deve ser um array de IDs",
                    code: 400
                });
            }

            for (let i = 0; i < data.destinatarios.length; i++) {
                const id = data.destinatarios[i];

                const fin = await this.getUmFinanceiro(id);

                if (!fin) continue;

                await sendMessage(data.mensagem, fin.celular); 
            }

            return resolve({
                success: true,
                message: "Todas as mensagens foram enviadas com sucesso",
                code: 200
            });

        } catch (error) {
            console.error("Erro ao enviar mensagem", error);
            return reject({ 
                error: true,
                message: "Erro ao enviar mensagem",
                code: 500
            });
        }
    });
}



    /**
     * Deleta uma loja
     * @param {*} id 
     * @returns 
     */
    delete(id) {
        return new Promise((resolve, reject) => {
            this.#db.query(
                "DELETE FROM lojas WHERE id = ?",
                [id],
                (error) => {
                    if (error) {
                        console.error("Erro ao deletar loja ", error);
                        return reject({ error: true, message: "Esta loja possui vínculos!", code: 409 });
                    }
                    return resolve({ success: true, message: "Loja deletada com sucesso!", code: 200 });
                }
            );
        });
    }

    /**
     * Atualiza uma loja
     * @param {*} data 
     * @returns 
     */
    update(data) {
        return new Promise(async (resolve, reject) => {
            try {
                const campos = [];
                const valores = [];

                // Atualiza apenas campos enviados
                if (data.razao_social) {
                    campos.push("razao_social = ?");
                    valores.push(data.razao_social);
                }
                if (data.email) {
                    campos.push("email = ?");
                    valores.push(data.email);
                }
                if (data.cep) {
                    campos.push("cep = ?");
                    valores.push(data.cep);
                }


                if (campos.length === 0) {
                    return reject({ error: true, message: "Nenhum dado para atualizar", code: 400 });
                }

                valores.push(data.id);

                const sql = `UPDATE lojas SET ${campos.join(", ")} WHERE id = ?`;

                this.#db.query(sql, valores, (error, response) => {
                    if (error) {
                        console.error(error);
                        return reject({ error: true, message: "Erro ao atualizar loja", code: 500 });
                    }
                    return resolve({ success: true, message: "Loja atualizada com sucesso!", code: 200 });
                });

            } catch (error) {
                console.error(error);
                return reject({ error: true, message: "Erro ao atualizar loja", code: 500 });
            }
        });
    }



    login(data) {
        return new Promise(async (resolve, reject) => {
            try {

                this.#db.query(
                    `SELECT * FROM users WHERE cnpj_cpf = ?`,
                    [data.cnpj_cpf],
                    async (error, response) => {
                        if (error) {
                            console.error("Erro ao buscar dados: ", error)
                            return reject({ error: true, message: "Erro ao buscar dados", code: 500 });
                        }
                        if (response.length === 0) {
                            return reject({ error: true, message: "Usuário não encontrado", code: 404 });
                        }
                        const userDB = response[0];

                        const isMatch = await bcrypt.compare(data.senha, userDB.senha);

                        if (!isMatch) {
                            return reject({ error: true, message: "Senha incorreta", code: 401 });
                        }

                        const token = await genereteToken(userDB);

                        return resolve({
                            token: token,
                            user: {
                                nome: userDB.nome,
                                id: userDB.trueId,
                                cnpj_cpf: userDB.cnpj_cpf,
                                email: userDB.email,
                                role: userDB.role,
                            }
                        });
                    }
                );
            } catch (err) {
                console.error("Login error: ", err);
                return reject({ error: true, message: "Erro ao fazer login", code: 500 });
            }
        });
    }


    /**
     * Realiza o logout do usuário
     * @param {*} data email do usuário
     * @returns mensagem referente ao resultado obtido
     */
    logout(auth) {
        return new Promise(async (resolve, reject) => {

            this.#db.query(
                "UPDATE user set auth=null WHERE auth=?",
                [auth],
                async (error, response) => {
                    if (error) {
                        console.error(error);
                        return reject({ error: true, message: "Erro ao realizar logout", code: 500 });
                    }
                    return resolve({ success: true, message: "Logout realizado com sucesso!", code: 200 });
                }
            );
        });
    }


    enviarMensagemParaResponsavel(data) {
        return new Promise(async (resolve, reject) => {
            try {


                if (!data.empresaId || !data.mensagem || !data.destinatarios) {
                    return reject({
                        error: true,
                        message: "Dados incompletos para envio de mensagem",
                        code: 400
                    });
                }

                for (const responsavel of data.destinatarios) {
                    const telefone = responsavel.telefone;

                    if (!telefone) {
                        console.warn(`Responsável ${responsavel.nome} sem telefone.`);
                        continue;
                    }

                    await sendMessage(data.mensagem, "55" + telefone);
                }

                return resolve({
                    success: true,
                    message: "Mensagens enviadas com sucesso.",
                    code: 200
                });

            } catch (error) {
                console.error("Erro ao enviar mensagem para responsáveis:", error);
                return reject({
                    error: true,
                    message: "Erro ao enviar mensagem para responsáveis",
                    code: 500
                });
            }
        });
    }


}

module.exports = AuthRepository;
