const { genereteToken } = require("../../config/auth/jwt");
const nodemailer = require("nodemailer");
const path = require("path");
const https = require("https");
const fs = require("fs");
const bcrypt = require("bcrypt");
const { validarCNPJ } = require("../../util/validarCNPJ");
const { validarCPF } = require("../../util/validarCPF");
const { sendMessage } = require("../../util/gzappy");

class AuthRepository {
  #db = require("../../config/db");

  constructor() {}

  /**
   *
   * @param {*} data
   * @returns
   */
  register(data) {
    return new Promise(async (resolve, reject) => {
      try {
        const resultado =
          data.cnpj_cpf.length === 11
            ? validarCPF(data.cnpj_cpf)
            : data.cnpj_cpf.length === 14
              ? validarCNPJ(data.cnpj_cpf)
              : null;

        if (!resultado) {
          return reject({
            error: true,
            message: "CNPJ ou CPF inválido!",
            code: 422,
          });
        }

        const hashedPassword = await bcrypt.hash(data.senha, 10);

        this.#db.query(
          `INSERT INTO admins (nome, cnpj_cpf, email,  senha, telefone) VALUES (?, ?, ?, ?, ?)`,
          [data.nome, data.cnpj_cpf, data.email, hashedPassword, data.telefone],
          async (error, response) => {
            if (error) {
              console.error(error);
              return reject({
                error: true,
                message: "Erro ao cadastrar admin",
                code: 500,
              });
            }

            this.#db.query(
              `INSERT INTO users (nome, cnpj_cpf, senha, role, trueId)
                             SELECT nome, cnpj_cpf, senha, 'admin', id
                             FROM admins
                             WHERE cnpj_cpf NOT IN (SELECT cnpj_cpf FROM users);`,
              (error) => {
                if (error) {
                  console.error(error);
                  return reject({
                    error: true,
                    message: "Erro ao cadastrar usuário admin",
                    code: 500,
                  });
                }

                this.#db.query(
                  `UPDATE admins a
                                     INNER JOIN users u ON a.cnpj_cpf = u.cnpj_cpf
                                     SET a.id_user = u.id
                                     WHERE u.role = 'admin';`,
                  (error) => {
                    if (error) {
                      console.error(error);
                      return reject({
                        error: true,
                        message: "Erro ao atualizar admin",
                        code: 500,
                      });
                    }
                  },
                );
              },
            );

            return resolve({
              success: true,
              message: "Admin cadastrado com sucesso!",
              code: 200,
            });
          },
        );
      } catch (err) {
        console.error("Erro ao cadastrar admin ", err);
        return reject({
          error: true,
          message: "Erro ao cadastrar admin",
          code: 500,
        });
      }
    });
  }

  registerEmpresa(data) {
    return new Promise(async (resolve, reject) => {
      try {
        const resultado =
          data.cnpj_cpf.length === 11
            ? validarCPF(data.cnpj_cpf)
            : data.cnpj_cpf.length === 14
              ? validarCNPJ(data.cnpj_cpf)
              : null;

        if (!resultado) {
          return reject({
            error: true,
            message: "CNPJ ou CPF inválido!",
            code: 422,
          });
        }

        this.#db.query(
          "SELECT id FROM empresas WHERE cnpj_cpf = ?",
          [data.cnpj_cpf],
          async (error, results) => {
            if (error) {
              console.error(error);
              return reject({
                error: true,
                message: "Erro ao verificar empresa",
                code: 500,
              });
            }

            if (results.length > 0) {
              return reject({
                error: true,
                message: "CNPJ ou CPF já cadastrado!",
                code: 409,
              });
            }

            const hashedPassword = await bcrypt.hash(data.senha_inicial, 10);

            this.#db.query(
              `
                        INSERT INTO empresas (
                            nome, cnpj_cpf, senha, telefone_empresa,
                            responsavel1_nome, responsavel1_celular, responsavel1_email,
                            responsavel2_nome, responsavel2_celular, responsavel2_email,
                            exige_foto_epi, contrato_periodo, contrato_valor,
                            contrato_pagamento, contrato_tipo, contrato_descricao,
                            whatsapp_notificacao_1, whatsapp_notificacao_2, atividade
                        )
                        VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
                        `,
              [
                data.razao_social,
                data.cnpj_cpf,
                hashedPassword,
                data.telefone,
                data.resp1_nome,
                data.resp1_celular,
                data.resp1_email,
                data.resp2_nome,
                data.resp2_celular,
                data.resp2_email,
                data.exige_epi,
                data.contrato_periodo,
                data.contrato_valor,
                data.contrato_pagamento,
                data.contrato_tipo,
                data.contrato_descricao,
                data.whatsapp_notificacao_1,
                data.whatsapp_notificacao_2,
                data.atividade,
              ],
              async (error, insertResult) => {
                if (error) {
                  console.error(error);
                  return reject({
                    error: true,
                    message: "Erro ao cadastrar empresa",
                    code: 500,
                  });
                }

                // INSERE O USER
                this.#db.query(
                  `
                                INSERT INTO users (nome, senha, role, cnpj_cpf, trueId)
                                SELECT nome, senha, 'empresa', cnpj_cpf, id
                                FROM empresas
                                WHERE cnpj_cpf = ? AND cnpj_cpf NOT IN (SELECT cnpj_cpf FROM users);
                                `,
                  [data.cnpj_cpf],
                  (error, insertUsers) => {
                    if (error) {
                      console.error(error);
                      return reject({
                        error: true,
                        message: "Erro ao cadastrar usuário da empresa",
                        code: 500,
                      });
                    }

                    // UPDATE EMPRESA COM ID DO USER
                    this.#db.query(
                      `
                                        UPDATE empresas a
                                        INNER JOIN users u ON a.cnpj_cpf = u.cnpj_cpf
                                        SET a.id_user = u.id
                                        WHERE u.role = 'empresa' AND a.cnpj_cpf = ?;
                                        `,
                      [data.cnpj_cpf],
                      (error, updateResult) => {
                        if (error) {
                          console.error(error);
                          return reject({
                            error: true,
                            message:
                              "Erro ao atualizar empresa com ID do usuário",
                            code: 500,
                          });
                        }

                        return resolve({
                          success: true,
                          message: "Empresa cadastrada com sucesso!",
                          code: 200,
                        });
                      },
                    );
                  },
                );
              },
            );
          },
        );
      } catch (err) {
        console.error("Erro ao cadastrar empresa ", err);
        return reject({
          error: true,
          message: "Erro ao cadastrar empresa",
          code: 500,
        });
      }
    });
  }

  updateEmpresa(data, id_empresa) {
    return new Promise(async (resolve, reject) => {
      try {
        const camposEmpresa = [];
        const valoresEmpresa = [];

        const camposUser = [];
        const valoresUser = [];

        // --- Atualização da senha para ambos ---
        if (data.senha_inicial) {
          const hash = await bcrypt.hash(data.senha_inicial, 10);

          // empresa.senha
          camposEmpresa.push("senha = ?");
          valoresEmpresa.push(hash);

          // users.senha
          camposUser.push("senha = ?");
          valoresUser.push(hash);
        }

        // Mapa de campos apenas para EMPRESAS
        const mapaDeCamposEmpresa = {
          razao_social: "nome",
          telefone: "telefone_empresa",
          resp1_nome: "responsavel1_nome",
          resp1_celular: "responsavel1_celular",
          resp1_email: "responsavel1_email",
          resp2_nome: "responsavel2_nome",
          resp2_celular: "responsavel2_celular",
          resp2_email: "responsavel2_email",
          exige_epi: "exige_foto_epi",
          contrato_periodo: "contrato_periodo",
          contrato_valor: "contrato_valor",
          contrato_pagamento: "contrato_pagamento",
          contrato_tipo: "contrato_tipo",
          contrato_descricao: "contrato_descricao",
          whatsapp_notificacao_1: "whatsapp_notificacao_1",
          whatsapp_notificacao_2: "whatsapp_notificacao_2",
          atividade: "atividade",
        };

        // Mapa de campos REALMENTE existentes em users
        const mapaDeCamposUser = {
          razao_social: "nome", // nome do responsável
          cnpj_cpf: "cnpj_cpf", // caso venha do request
        };

        // ----- Preenche campos da empresa -----
        Object.keys(mapaDeCamposEmpresa).forEach((campoData) => {
          if (data[campoData] !== undefined) {
            camposEmpresa.push(`${mapaDeCamposEmpresa[campoData]} = ?`);
            valoresEmpresa.push(data[campoData]);
          }
        });

        // ----- Preenche apenas campos permitidos em users -----
        Object.keys(mapaDeCamposUser).forEach((campoData) => {
          if (data[campoData] !== undefined) {
            camposUser.push(`${mapaDeCamposUser[campoData]} = ?`);
            valoresUser.push(data[campoData]);
          }
        });

        if (camposEmpresa.length === 0 && camposUser.length === 0) {
          return reject({
            error: true,
            message: "Nenhum dado válido enviado para atualização.",
            code: 400,
          });
        }

        // ------------------ UPDATE EMPRESA ------------------
        const sqlEmpresa = `UPDATE empresas SET ${camposEmpresa.join(", ")} WHERE id = ?`;
        valoresEmpresa.push(id_empresa);

        this.#db.query(sqlEmpresa, valoresEmpresa, (error, response) => {
          if (error) {
            console.error("Erro SQL Update Empresa:", error);
            return reject({
              error: true,
              message: "Erro interno ao atualizar empresa",
              code: 500,
            });
          }

          if (response.affectedRows === 0) {
            return reject({
              error: true,
              message: "Empresa não encontrada.",
              code: 404,
            });
          }

          // ------------------ UPDATE USER ------------------
          if (camposUser.length === 0) {
            return resolve({
              success: true,
              message: "Empresa atualizada com sucesso!",
              code: 200,
            });
          }

          const sqlUser = `UPDATE users SET ${camposUser.join(", ")} WHERE trueId = ? AND role = 'empresa'`;
          valoresUser.push(id_empresa);

          this.#db.query(sqlUser, valoresUser, (errorUser) => {
            if (errorUser) {
              console.error("Erro SQL Update User:", errorUser);
              return reject({
                error: true,
                message: "Erro ao atualizar usuário vinculado",
                code: 500,
              });
            }

            return resolve({
              success: true,
              message: "Empresa e usuário atualizados com sucesso!",
              code: 200,
            });
          });
        });
      } catch (err) {
        console.error("Erro no updateEmpresa:", err);
        return reject({
          error: true,
          message: "Erro crítico ao atualizar.",
          code: 500,
        });
      }
    });
  }

  deleteEmpresa(id_empresa) {
    return new Promise((resolve, reject) => {
      try {
        this.#db.query(
          `UPDATE empresas SET status = 'inativo' WHERE id = ?`,
          [id_empresa],
          (error, response) => {
            if (error) {
              console.error("Erro ao deletar empresa");
              return reject({
                error: true,
                message: "Erro ao desativar empresa",
                code: 500,
              });
            }
            return resolve({
              success: true,
              message: "Empresa desativada com sucesso!",
              code: 200,
            });
          },
        );
      } catch (error) {
        console.error("Erro ao deletar empresa ", error);
        return reject({
          error: true,
          message: "Erro ao desativar empresa",
          code: 500,
        });
      }
    });
  }

  getEmpresas() {
    return new Promise((resolve, reject) => {
      try {
        this.#db.query(
          `SELECT 
                        e.*,
                        (SELECT JSON_ARRAYAGG(JSON_OBJECT('id', o.id, 'valor', o.valor, 'descricao', o.descricao, 'validade_proposta', o.validade_proposta, 'forma_pagamento', o.forma_pagamento, 'periodo_execucao', o.periodo_execucao, 'urls', o.urls, 'criado_em', o.criado_em, 'atualizado_em', o.atualizado_em, 'status', o.status, 'justificativa', o.justificativa, 'titulo', o.titulo)) 
                         FROM orcamentos o 
                         WHERE o.id_user = e.id) as orcamentos
                    FROM empresas e`,
          [],
          (error, empresas) => {
            if (error) {
              console.error("Erro ao buscar empresas ", error);
              return reject({
                error: true,
                message: "Erro ao buscar empresas",
                code: 500,
              });
            }

            // O resultado de orcamentos é uma string JSON, vamos fazer o parse.
            const empresasComOrcamentos = empresas.map((empresa) => {
              if (empresa.orcamentos) {
                empresa.orcamentos = JSON.parse(empresa.orcamentos);
              }
              return empresa;
            });
            return resolve(empresasComOrcamentos);
          },
        );
      } catch (error) {
        console.error("Erro ao buscar empresas ", error);
        return reject({
          error: true,
          message: "Erro ao buscar empresas",
          code: 500,
        });
      }
    });
  }

  getStatusEmpresa(id_empresa) {
    return new Promise((resolve, reject) => {
      try {
        this.#db.query(
          `SELECT status FROM empresas WHERE id = ?`,
          [id_empresa],
          (error, result) => {
            if (error) {
              console.error("Erro ao buscar status ", error);
              return reject({
                error: true,
                message: "Erro ao buscar status",
                code: 500,
              });
            }
            if (result.length === 0) {
              return reject({
                error: true,
                message: "Empresa não encontrada.",
                code: 404,
              });
            }
            return resolve(result[0]);
          },
        );
      } catch (error) {
        console.error("Erro ao buscar status da empresa ", error);
        return reject({
          error: true,
          message: "Erro ao buscar status da empresa",
          code: 500,
        });
      }
    });
  }

  /**
   * Lista todas as lojas
   */
  getLojas() {
    return new Promise((resolve, reject) => {
      try {
        this.#db.query(
          `SELECT id, razao_social, documento, email, cep
           FROM lojas`,
          [],
          (error, lojas) => {
            if (error) {
              console.error("Erro ao buscar lojas ", error);
              return reject({
                error: true,
                message: "Erro ao buscar lojas",
                code: 500,
              });
            }
            return resolve(lojas);
          },
        );
      } catch (error) {
        console.error("Erro ao buscar lojas ", error);
        return reject({
          error: true,
          message: "Erro ao buscar lojas",
          code: 500,
        });
      }
    });
  }

  /**
   * Busca apenas uma loja
   * @param {*} id
   * @returns
   */
  getUmaLoja(id) {
    return new Promise((resolve, reject) => {
      try {
        this.#db.query(
          `SELECT id, razao_social, documento, email, cep 
           FROM lojas 
           WHERE id = ?`,
          [id],
          (error, loja) => {
            if (error) {
              console.error("Erro ao buscar loja ", error);
              return reject({
                error: true,
                message: "Erro ao buscar loja",
                code: 500,
              });
            }
            if (loja.length === 0) {
              return resolve({
                error: true,
                message: "Nenhuma loja encontrada com esse ID",
                code: 404,
              });
            }
            return resolve(loja[0]);
          },
        );
      } catch (error) {
        console.error("Erro ao buscar loja ", error);
        return reject({
          error: true,
          message: "Erro ao buscar loja",
          code: 500,
        });
      }
    });
  }

  /**
   * Deleta uma loja
   * @param {*} id
   * @returns
   */
  delete(id) {
    return new Promise((resolve, reject) => {
      this.#db.query("DELETE FROM lojas WHERE id = ?", [id], (error) => {
        if (error) {
          console.error("Erro ao deletar loja ", error);
          return reject({
            error: true,
            message: "Esta loja possui vínculos!",
            code: 409,
          });
        }
        return resolve({
          success: true,
          message: "Loja deletada com sucesso!",
          code: 200,
        });
      });
    });
  }

  /**
   * Atualiza uma loja
   * @param {*} data
   * @returns
   */
  update(data) {
    return new Promise(async (resolve, reject) => {
      try {
        const campos = [];
        const valores = [];

        // Atualiza apenas campos enviados
        if (data.razao_social) {
          campos.push("razao_social = ?");
          valores.push(data.razao_social);
        }
        if (data.email) {
          campos.push("email = ?");
          valores.push(data.email);
        }
        if (data.cep) {
          campos.push("cep = ?");
          valores.push(data.cep);
        }

        if (campos.length === 0) {
          return reject({
            error: true,
            message: "Nenhum dado para atualizar",
            code: 400,
          });
        }

        valores.push(data.id);

        const sql = `UPDATE lojas SET ${campos.join(", ")} WHERE id = ?`;

        this.#db.query(sql, valores, (error, response) => {
          if (error) {
            console.error(error);
            return reject({
              error: true,
              message: "Erro ao atualizar loja",
              code: 500,
            });
          }
          return resolve({
            success: true,
            message: "Loja atualizada com sucesso!",
            code: 200,
          });
        });
      } catch (error) {
        console.error(error);
        return reject({
          error: true,
          message: "Erro ao atualizar loja",
          code: 500,
        });
      }
    });
  }

  // login(data) {
  //     return new Promise(async (resolve, reject) => {

  //         try {
  //             this.#db.query(
  //                 "SELECT * FROM user WHERE documento=?",
  //                 [data.documento],
  //                 async (error, response) => {
  //                     if (error) {
  //                         console.error("Erro ao buscar dados: ", error);
  //                         return reject({ error: true, message: "Erro ao buscar dados", code: 500 });
  //                     }

  //                     if (response.length === 0) {
  //                         return reject({ error: true, message: "Usuário não encontrado", code: 404 });
  //                     }

  //                     const userDB = response[0];

  //                     const isMatch = await bcrypt.compare(data.senha, userDB.senha);

  //                     if (!isMatch) {
  //                         return reject({ error: true, message: "Senha incorreta", code: 401 });
  //                     }

  //                     this.#db.query("UPDATE usuarios SET auth=null WHERE email=?",[data.email],
  //                     async (updateError) => {
  //                     if (updateError) {
  //                         console.error("Database update error:", updateError);
  //                         return reject({ error: true, message: "Erro ao atualizar login", code: 500 });
  //                     }
  //                     });

  //                     const token = await genereteToken(userDB);

  //                     this.#db.query(
  //                         "UPDATE user SET auth=? WHERE documento=?",
  //                         [token, data.documento],
  //                         (updateError) => {
  //                             if (updateError) {
  //                                 console.error("Database update error:", updateError);
  //                                 return reject({ error: true, message: "Erro ao realizar login", code: 500 });
  //                             }

  //                             return resolve({
  //                                 token: token,
  //                                 user: {
  //                                     id: userDB.id,
  //                                     razao_social: userDB.razao_social,
  //                                     cnpj_cpf: userDB.cnpj_cpf,
  //                                     email: userDB.email,
  //                                     role: userDB.role,
  //                                     auth: token
  //                                 }
  //                             });
  //                         });
  //                 }
  //             );
  //         } catch (err) {
  //             console.error("Login error: ", err);
  //             return reject({ error: true, message: "Erro ao fazer login", code: 500 });
  //         }

  //     });
  // }

  login(data) {
    return new Promise(async (resolve, reject) => {
      try {
        this.#db.query(
          `SELECT * FROM users WHERE cnpj_cpf = ?`,
          [data.cnpj_cpf],
          async (error, response) => {
            if (error) {
              return reject({
                error: true,
                message: "Erro ao buscar dados",
                code: 500,
              });
            }

            if (response.length === 0) {
              return reject({
                error: true,
                message: "Usuário não encontrado",
                code: 404,
              });
            }

            const usersAutenticados = [];

            for (const user of response) {
              const isMatch = await bcrypt.compare(data.senha, user.senha);

              if (isMatch) {
                const token = await genereteToken(user);

                usersAutenticados.push({
                  token,
                  user: {
                    id: user.trueId,
                    nome: user.nome,
                    email: user.email,
                    role: user.role,
                    cnpj_cpf: user.cnpj_cpf,
                  },
                });
              }
            }

            if (usersAutenticados.length === 0) {
              return reject({
                error: true,
                message: "Senha inválida",
                code: 401,
              });
            }

            return resolve({
              usersAutenticados,
            });
          },
        );
      } catch (err) {
        return reject(err);
      }
    });
  }

  /**
   * Realiza o logout do usuário
   * @param {*} data email do usuário
   * @returns mensagem referente ao resultado obtido
   */
  logout(auth) {
    return new Promise(async (resolve, reject) => {
      this.#db.query(
        "UPDATE user set auth=null WHERE auth=?",
        [auth],
        async (error, response) => {
          if (error) {
            console.error(error);
            return reject({
              error: true,
              message: "Erro ao realizar logout",
              code: 500,
            });
          }
          return resolve({
            success: true,
            message: "Logout realizado com sucesso!",
            code: 200,
          });
        },
      );
    });
  }

  enviarMensagemParaResponsavel(data) {
    return new Promise(async (resolve, reject) => {
      try {
        if (!data.empresaId || !data.mensagem || !data.destinatarios) {
          return reject({
            error: true,
            message: "Dados incompletos para envio de mensagem",
            code: 400,
          });
        }

        for (const responsavel of data.destinatarios) {
          const telefone = responsavel.telefone;

          if (!telefone) {
            console.warn(`Responsável ${responsavel.nome} sem telefone.`);
            continue;
          }

          await sendMessage(data.mensagem, "55" + telefone);
        }

        return resolve({
          success: true,
          message: "Mensagens enviadas com sucesso.",
          code: 200,
        });
      } catch (error) {
        console.error("Erro ao enviar mensagem para responsáveis:", error);
        return reject({
          error: true,
          message: "Erro ao enviar mensagem para responsáveis",
          code: 500,
        });
      }
    });
  }
}

module.exports = AuthRepository;
